<?php
/**
 * Plugin Name: Honeypot Guard – Silent Anti-Spam
 * Plugin URI: https://www.mantasdigital.com/honeypot-guard/
 * Description: Anti-spam protection for forms, signups, and comments using advanced honeypot techniques. No CAPTCHAs, no user friction. Works with Contact Form 7, WPForms, Gravity Forms, Ninja Forms, WooCommerce, bbPress, and more.
 * Version: 2.1.0
 * Author: MantasDigital
 * Author URI: https://www.mantasdigital.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: honeypot-guard-silent-anti-spam
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

// Security: Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Security: Prevent direct file access
if (!defined('WPINC')) {
    die;
}

define('HONEYPOT_GUARD_VERSION', '2.1.0');
define('HONEYPOT_GUARD_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('HONEYPOT_GUARD_PLUGIN_URL', plugin_dir_url(__FILE__));
define('HONEYPOT_GUARD_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Load translations system
require_once HONEYPOT_GUARD_PLUGIN_DIR . 'includes/translations.php';

/**
 * Main HoneypotGuard Class
 *
 * @since 1.0.0
 */
final class HoneypotGuard {

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Database table name
     */
    private $table_name;

    /**
     * Blacklist table name
     */
    private $blacklist_table;

    /**
     * Plugin options (cached)
     */
    private $options;

    /**
     * Options cache key
     */
    private $options_cache_key = 'honeypot_guard_options_cache';

    /**
     * Cached honeypot field name
     */
    private $cached_honeypot_name = null;

    /**
     * Cached blacklist entries
     */
    private $cached_blacklist = null;

    /**
     * Page has forms flag (for lazy loading)
     */
    private $page_has_forms = null;

    /**
     * Realistic honeypot field names - expanded pool for dynamic rotation
     */
    private $honeypot_names = [
        // Contact/personal fields
        'middle_name',
        'maiden_name',
        'nickname_field',
        'preferred_name',
        'title_prefix',

        // Business fields
        'company_extension',
        'department_code',
        'employee_id',
        'cost_center',
        'purchase_order',

        // Communication fields
        'secondary_phone',
        'alt_email_address',
        'fax_number',
        'pager_number',
        'backup_contact',

        // Address fields
        'address_line_3',
        'suite_number',
        'mail_stop',
        'delivery_instructions',
        'building_code',

        // Misc fields
        'referral_source_detail',
        'how_heard_other',
        'promo_code_backup',
        'internal_reference',
        'legacy_account_id',
        'user_verify_id',
        'extra_billing_note',
        'preferred_contact_time',
        'additional_notes_field',
        'survey_feedback',
    ];

    /**
     * Rate limit tracking
     */
    private $rate_limit_transient = 'honeypot_guard_rate_limit_';

    /**
     * Project URLs
     */
    const LANDING_PAGE_URL = 'https://www.mantasdigital.com/honeypot-guard/';
    const SUPPORT_URL = 'https://www.mantasdigital.com/';

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'honeypot_guard_logs';
        $this->blacklist_table = $wpdb->prefix . 'honeypot_guard_blacklist';

        // Load options with caching for performance
        $this->load_options();

        // Core hooks
        add_action('plugins_loaded', [$this, 'init']);
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
        register_uninstall_hook(__FILE__, ['HoneypotGuard', 'uninstall']);
    }

    /**
     * Load options with object caching support
     */
    private function load_options() {
        // Try object cache first
        $cached = wp_cache_get($this->options_cache_key, 'honeypot_guard');
        if ($cached !== false) {
            $this->options = $cached;
            return;
        }

        // Load from database
        $this->options = get_option('honeypot_guard_options', $this->get_default_options());

        // Store in object cache (expires in 5 minutes)
        wp_cache_set($this->options_cache_key, $this->options, 'honeypot_guard', 300);
    }

    /**
     * Clear options cache (called on settings save)
     */
    public function clear_options_cache() {
        wp_cache_delete($this->options_cache_key, 'honeypot_guard');
        $this->cached_honeypot_name = null;
        $this->cached_blacklist = null;
    }

    /**
     * Extract only the relevant POST fields for spam analysis
     * Avoids processing the entire $_POST array
     *
     * @return array Selected POST fields
     */
    private function get_relevant_post_fields() {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified by form plugin or our own check
        $post_data = $_POST;

        // Define the keys we actually need for spam analysis
        $honeypot_field = $this->get_honeypot_field_name();
        $relevant_keys = [
            // Our own fields
            '_oh_timestamp', '_oh_nonce',
            // Honeypot field(s)
            $honeypot_field,
            // Common form fields we analyze
            'email', 'e-mail', 'your-email', 'mail', 'email_address',
            'name', 'your-name', 'first_name', 'last_name', 'first', 'last',
            'subject', 'your-subject',
            'message', 'your-message', 'comment', 'content', 'body', 'textarea', 'description',
            'url', 'website', 'homepage', 'link',
            'company', 'phone', 'tel', 'address',
            // CF7 internal fields
            '_wpcf7', '_wpcf7_unit_tag', '_wpcf7_version',
            // WPForms / GF / NF identifiers
            'wpforms', 'gform_submit', '_ninja_forms_display_submit',
        ];

        // Also include multiple honeypot fields if enabled
        if (!empty($this->options['honeypot_multiple'])) {
            $extra_names = $this->get_honeypot_field_names(2);
            $relevant_keys = array_merge($relevant_keys, $extra_names);
        }

        $selected = [];
        foreach ($relevant_keys as $key) {
            if (isset($post_data[$key])) {
                $selected[$key] = $post_data[$key];
            }
        }

        // Also include any keys that match common form field patterns (for custom forms)
        foreach ($post_data as $key => $value) {
            if (isset($selected[$key])) {
                continue;
            }
            // Include fields matching common patterns used for analysis
            if (preg_match('/^(email|name|subject|message|comment|content|body|url|website|phone|company|address|text|titel|betreff|nachricht|vardas|pavarde|zinute|nom|prenom|sujet|nombre|apellido|asunto|mensaje)/i', $key)) {
                $selected[$key] = $value;
            }
            // Include honeypot-related internal fields
            if (strpos($key, '_oh_') === 0) {
                $selected[$key] = $value;
            }
        }

        return $selected;
    }

    /**
     * Sanitize POST data array for spam analysis
     * Applies appropriate sanitization to each field while preserving data for analysis
     *
     * @param array $data Raw POST data array
     * @return array Sanitized data array
     */
    public function sanitize_submission_data($data) {
        if (!is_array($data)) {
            return [];
        }

        $sanitized = [];
        foreach ($data as $key => $value) {
            // Sanitize the key
            $safe_key = sanitize_key($key);

            // Handle arrays recursively
            if (is_array($value)) {
                $sanitized[$safe_key] = $this->sanitize_submission_data($value);
                continue;
            }

            // Unslash and sanitize based on field type
            $value = wp_unslash($value);

            // Email fields
            if (preg_match('/email|e-mail|your-email/i', $key)) {
                $sanitized[$safe_key] = sanitize_email($value);
            }
            // URL fields
            elseif (preg_match('/url|website|homepage|link/i', $key)) {
                $sanitized[$safe_key] = esc_url_raw($value);
            }
            // Textarea/message fields - use textarea sanitization to preserve newlines
            elseif (preg_match('/message|comment|content|body|textarea|description|your-message/i', $key)) {
                $sanitized[$safe_key] = sanitize_textarea_field($value);
            }
            // Internal honeypot/timestamp fields
            elseif (strpos($key, '_oh_') === 0 || strpos($key, '_wpcf7') === 0) {
                $sanitized[$safe_key] = sanitize_text_field($value);
            }
            // Default: text field sanitization
            else {
                $sanitized[$safe_key] = sanitize_text_field($value);
            }
        }

        return $sanitized;
    }

    /**
     * Prevent cloning
     */
    private function __clone() {}

    /**
     * Prevent unserialization
     */
    public function __wakeup() {
        throw new Exception('Cannot unserialize singleton');
    }

    /**
     * Default options
     */
    public function get_default_options() {
        return [
            // Language
            'plugin_language' => 'en',

            // Honeypot
            'honeypot_enabled' => true,
            'honeypot_field_name' => 'middle_name',
            'honeypot_mode' => 'auto', // auto, custom, dynamic
            'honeypot_rotation' => 'session', // session, daily, hourly
            'honeypot_inject_js' => true, // Inject via JS (harder for bots to detect)
            'honeypot_multiple' => false, // Add multiple honeypot fields
            'honeypot_custom_names' => '', // User's custom field names (one per line)

            // Timestamp
            'timestamp_enabled' => true,
            'min_submission_time' => 2,
            'max_submission_time' => 3600,

            // Character Filters
            'cyrillic_filter' => false,
            'hanzi_filter' => false,
            'bengali_filter' => false,
            'latin_filter' => false,

            // Blacklists (granular)
            'blacklist_emails' => '',
            'blacklist_domains' => '',
            'blacklist_keywords_subject' => '',
            'blacklist_keywords_body' => '',
            'blacklist_keywords_all' => '',
            'blacklist_ips' => '',

            // Heuristics
            'heuristic_enabled' => true,
            'max_links' => 3,
            'spam_threshold' => 70,

            // Rate Limiting
            'rate_limit_enabled' => true,
            'rate_limit_submissions' => 5,
            'rate_limit_window' => 60,

            // Security
            'block_tor_exit_nodes' => false,
            'block_vpn_datacenter' => false,
            'require_referer' => true,
            'validate_honeypot_js' => true,
            'cookie_check_enabled' => false,

            // Logging
            'log_retention_days' => 30,
            'log_clean_submissions' => false,

            // Escalation
            'escalation_threshold' => 3,
            'escalation_window' => 60,

            // AI Gibberish Detection
            'gibberish_detection_enabled' => false,
            'gibberish_ai_provider' => 'openai', // openai, anthropic
            'gibberish_api_key' => '',
            'gibberish_language' => 'auto', // 'auto' detects from WPML/Polylang/site language
            'gibberish_domain_languages' => '', // Domain-specific languages (domain:lang per line)
            'gibberish_fields' => [
                // English
                'name', 'first', 'last', 'subject', 'message', 'comment', 'company', 'website', 'title', 'text', 'body', 'content',
                // Lithuanian
                'vardas', 'pavarde', 'pavadinimas', 'zinute', 'pranesimas', 'komentaras', 'imone', 'svetaine', 'tema',
                // German
                'vorname', 'nachname', 'betreff', 'nachricht', 'kommentar', 'firma', 'webseite',
                // French
                'nom', 'prenom', 'sujet', 'commentaire', 'entreprise', 'societe',
                // Spanish
                'nombre', 'apellido', 'asunto', 'mensaje', 'comentario', 'empresa'
            ],
            'gibberish_min_length' => 5, // Minimum text length to check
            'gibberish_score' => 85, // Spam score for gibberish detection

            // Spam Messages
            'spam_message_mode' => 'auto', // 'auto' (detect language), 'custom' (single custom), 'per_domain' (per domain)
            'custom_spam_message' => '',
            'spam_messages_per_domain' => '', // Format: domain|message per line
            'spam_message_default' => '', // Default message when no domain match

            // Advanced
            'whitelist_logged_in' => true,
            'whitelist_roles' => ['administrator', 'editor'],

            // Registration Form Protection (unchecked by default)
            'protect_registration_forms' => false,
        ];
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Note: Textdomain is loaded automatically by WordPress.org for hosted plugins (since WP 4.6)

        // Ensure database tables exist (in case activation didn't run)
        $this->maybe_create_tables();

        // Admin hooks - only load on admin pages
        if (is_admin()) {
            add_action('admin_menu', [$this, 'add_admin_menu']);
            add_action('admin_init', [$this, 'register_settings']);
            add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
            add_action('wp_ajax_honeypot_guard_delete_log', [$this, 'ajax_delete_log']);
            add_action('wp_ajax_honeypot_guard_clear_logs', [$this, 'ajax_clear_logs']);
            add_action('wp_ajax_honeypot_guard_get_stats', [$this, 'ajax_get_stats']);
            add_action('wp_ajax_honeypot_guard_add_blacklist', [$this, 'ajax_add_blacklist']);
            add_action('wp_ajax_honeypot_guard_remove_blacklist', [$this, 'ajax_remove_blacklist']);
            add_action('wp_ajax_honeypot_guard_remove_blacklist_by_value', [$this, 'ajax_remove_blacklist_by_value']);
            add_action('wp_ajax_honeypot_guard_test_submission', [$this, 'ajax_test_submission']);
            add_filter('plugin_action_links_' . HONEYPOT_GUARD_PLUGIN_BASENAME, [$this, 'add_plugin_links']);

            // Clear cache on settings update
            add_action('update_option_honeypot_guard_options', [$this, 'clear_options_cache']);
        }

        // Frontend hooks - only load if honeypot is enabled and not admin
        if (!is_admin() && $this->options['honeypot_enabled']) {
            // Enqueue fbq interceptor and honeypot protection scripts via wp_enqueue_scripts
            add_action('wp_enqueue_scripts', [$this, 'enqueue_frontend_scripts'], 1);
        }

        // Contact Form 7 Integration - only if CF7 is active
        if (class_exists('WPCF7')) {
            // Use late priority to let other spam checks run first
            add_filter('wpcf7_spam', [$this, 'check_cf7_spam'], 99, 2);
            add_action('wpcf7_init', [$this, 'add_cf7_honeypot_field']);
            add_filter('wpcf7_form_hidden_fields', [$this, 'add_cf7_hidden_fields']);
            // Log successful submissions too
            add_action('wpcf7_mail_sent', [$this, 'log_cf7_success']);
            // Skip validation hook - just log, don't block
            add_filter('wpcf7_validate', [$this, 'cf7_validate_log'], 99, 2);
            // Customize spam message with translation
            add_filter('wpcf7_display_message', [$this, 'customize_cf7_spam_message'], 10, 2);
            // Ensure spam message appears in AJAX response
            add_filter('wpcf7_ajax_json_echo', [$this, 'customize_cf7_ajax_response'], 10, 2);
        }

        // WPForms Integration - only if WPForms is active
        if (class_exists('WPForms')) {
            add_filter('wpforms_process_before_form_data', [$this, 'check_wpforms_spam'], 10, 2);
        }

        // Gravity Forms Integration - only if GF is active
        if (class_exists('GFForms')) {
            add_filter('gform_entry_is_spam', [$this, 'check_gravity_forms_spam'], 10, 3);
        }

        // Ninja Forms Integration
        if (class_exists('Ninja_Forms')) {
            add_filter('ninja_forms_submit_data', [$this, 'check_ninja_forms_spam'], 10, 1);
        }

        // WordPress Comments Integration
        add_filter('preprocess_comment', [$this, 'check_comment_spam'], 10, 1);

        // bbPress Forum Integration
        if (class_exists('bbPress')) {
            add_filter('bbp_new_topic_pre_content', [$this, 'check_bbpress_spam'], 10, 1);
            add_filter('bbp_new_reply_pre_content', [$this, 'check_bbpress_spam'], 10, 1);
        }

        // Registration Form Protection (only if enabled)
        if (!empty($this->options['protect_registration_forms'])) {
            // WordPress Registration
            add_action('register_form', [$this, 'add_registration_honeypot_fields']);
            add_filter('registration_errors', [$this, 'check_registration_spam'], 10, 3);
            // Fire conversion events after successful user creation
            add_action('user_register', [$this, 'on_user_registered'], 10, 2);

            // WooCommerce Registration
            if (class_exists('WooCommerce')) {
                add_action('woocommerce_register_form', [$this, 'add_registration_honeypot_fields']);
                add_filter('woocommerce_registration_errors', [$this, 'check_woocommerce_registration_spam'], 10, 3);
                // Also protect WooCommerce checkout registration
                add_action('woocommerce_after_checkout_registration_form', [$this, 'add_registration_honeypot_fields']);
                // Fire conversion events after WooCommerce customer creation
                add_action('woocommerce_created_customer', [$this, 'on_woocommerce_customer_created'], 10, 3);
            }

            // Output registration conversion events script via enqueue
            add_action('wp_enqueue_scripts', [$this, 'enqueue_registration_conversion_script'], 20);
        }

        // Generic form handling - only on POST requests
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && !is_admin()) {
            add_action('init', [$this, 'check_generic_form_submission'], 1);
        }

        // REST API - lazy registration
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Cleanup cron - only schedule if not already scheduled
        if (!wp_next_scheduled('honeypot_guard_cleanup_logs')) {
            wp_schedule_event(time(), 'daily', 'honeypot_guard_cleanup_logs');
        }
        add_action('honeypot_guard_cleanup_logs', [$this, 'cleanup_old_logs']);

        // Security headers - only on plugin admin pages
        add_action('send_headers', [$this, 'add_security_headers']);
    }

    /**
     * Plugin activation
     */
    public function activate() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Logs table
        $sql_logs = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            status varchar(20) NOT NULL DEFAULT 'spam',
            reason text NOT NULL,
            content longtext NOT NULL,
            ip_address varchar(45) DEFAULT NULL,
            user_agent varchar(500) DEFAULT NULL,
            form_source varchar(100) DEFAULT NULL,
            source_url varchar(500) DEFAULT NULL,
            spam_score smallint(3) UNSIGNED DEFAULT 0,
            email_field varchar(255) DEFAULT NULL,
            subject_field varchar(255) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status (status),
            KEY created_at (created_at),
            KEY ip_address (ip_address),
            KEY email_field (email_field)
        ) $charset_collate;";

        // Blacklist table
        $sql_blacklist = "CREATE TABLE IF NOT EXISTS {$this->blacklist_table} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            type varchar(20) NOT NULL,
            value varchar(255) NOT NULL,
            reason varchar(255) DEFAULT NULL,
            hits int(10) UNSIGNED DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            created_by bigint(20) UNSIGNED DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY type_value (type, value),
            KEY type (type),
            KEY hits (hits)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_logs);
        dbDelta($sql_blacklist);

        // Set default options
        if (!get_option('honeypot_guard_options')) {
            add_option('honeypot_guard_options', $this->get_default_options());
        }

        add_option('honeypot_guard_db_version', HONEYPOT_GUARD_VERSION);
        add_option('honeypot_guard_installed_at', current_time('mysql'));

        // Clear rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Ensure database tables exist and are up to date (runs on init)
     */
    private function maybe_create_tables() {
        global $wpdb;

        // Check if tables exist
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking table existence
        $logs_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_name}'") === $this->table_name;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking table existence
        $blacklist_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->blacklist_table}'") === $this->blacklist_table;

        // Create tables if missing
        if (!$logs_exists || !$blacklist_exists) {
            $this->activate();
        }

        // Check for column updates (source_url column added in 2.0.1)
        if ($logs_exists) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Schema check
            $column_exists = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'source_url'",
                    DB_NAME,
                    $this->table_name
                )
            );
            if (!$column_exists) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table schema migration
                $wpdb->query("ALTER TABLE {$this->table_name} ADD COLUMN source_url varchar(500) DEFAULT NULL AFTER form_source");
            }
        }
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        wp_clear_scheduled_hook('honeypot_guard_cleanup_logs');
        flush_rewrite_rules();
    }

    /**
     * Plugin uninstall
     */
    public static function uninstall() {
        global $wpdb;

        // Only delete data if option is set
        $options = get_option('honeypot_guard_options', []);
        if (!empty($options['delete_data_on_uninstall'])) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Uninstall cleanup
            $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}honeypot_guard_logs");
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Uninstall cleanup
            $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}honeypot_guard_blacklist");
            delete_option('honeypot_guard_options');
            delete_option('honeypot_guard_db_version');
            delete_option('honeypot_guard_installed_at');
        }

        // Clear transients
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Transient cleanup
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_honeypot_guard_rate_limit_%'");
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Transient cleanup
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_honeypot_guard_rate_limit_%'");
    }

    /**
     * Add security headers
     */
    public function add_security_headers() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce not needed for page check
        if (is_admin() && isset($_GET['page']) && strpos(sanitize_text_field(wp_unslash($_GET['page'])), 'honeypot-guard') !== false) {
            header('X-Content-Type-Options: nosniff');
            header('X-Frame-Options: SAMEORIGIN');
            header('X-XSS-Protection: 1; mode=block');
            header('Referrer-Policy: strict-origin-when-cross-origin');
        }
    }

    /**
     * =========================================================================
     * CORE SPAM DETECTION ENGINE
     * =========================================================================
     */

    /**
     * Analyze submission for spam
     */
    public function analyze_submission($data, $source = 'generic') {
        $result = [
            'is_spam' => false,
            'score' => 0,
            'reasons' => [],
            'details' => []
        ];

        // Skip for whitelisted users (but still log for testing visibility)
        if ($this->is_whitelisted_user()) {
            $result['details']['whitelisted'] = true;
            // Log whitelisted submissions so admin can see them
            if ($this->options['log_clean_submissions']) {
                $result['reasons'][] = 'Whitelisted user (spam checks skipped)';
                $this->log_submission($result, $data, $source, $this->extract_email_from_data($data), $this->extract_subject_from_data($data));
            }
            return $result;
        }

        $ip = $this->get_client_ip();

        // 1. IP Blacklist Check
        if ($this->is_ip_blacklisted($ip)) {
            $result['score'] += 100;
            $result['reasons'][] = 'IP address blacklisted';
            $result['details']['ip_blacklisted'] = true;
            $this->increment_blacklist_hits('ip', $ip);
        }

        // 2. Rate Limiting
        if ($this->options['rate_limit_enabled'] && $this->is_rate_limited($ip)) {
            $result['score'] += 100;
            $result['reasons'][] = 'Rate limit exceeded';
            $result['details']['rate_limited'] = true;
            // Auto-add IP to blacklist when rate limit is exceeded
            $this->add_rate_limited_ip_to_blacklist($ip);
        }

        // 3. Referer Check
        if ($this->options['require_referer'] && (!isset($_SERVER['HTTP_REFERER']) || empty($_SERVER['HTTP_REFERER']))) {
            $result['score'] += 30;
            $result['reasons'][] = 'Missing referer header';
            $result['details']['no_referer'] = true;
        }

        // 3b. Cookie Check - verifies JavaScript executed in a real browser
        if (!empty($this->options['cookie_check_enabled']) && empty($_COOKIE['_oh_verify'])) {
            $result['score'] += 40;
            $result['reasons'][] = 'Missing verification cookie';
            $result['details']['no_cookie'] = true;
        }

        // 4. Honeypot Check
        if ($this->options['honeypot_enabled']) {
            $honeypot_field = $this->get_honeypot_field_name();
            if (isset($data[$honeypot_field]) && !empty(trim($data[$honeypot_field]))) {
                $result['score'] += 100;
                $result['reasons'][] = 'Honeypot field triggered';
                $result['details']['honeypot'] = true;
            }
        }

        // 5. Timestamp Check
        if ($this->options['timestamp_enabled']) {
            $timestamp = isset($data['_oh_timestamp']) ? intval($data['_oh_timestamp']) : 0;
            if ($timestamp > 0) {
                $elapsed = time() - $timestamp;
                if ($elapsed < $this->options['min_submission_time']) {
                    $result['score'] += 80;
                    $result['reasons'][] = "Submitted too fast ({$elapsed}s)";
                    $result['details']['too_fast'] = true;
                }
                if ($elapsed > $this->options['max_submission_time']) {
                    $result['score'] += 40;
                    $result['reasons'][] = 'Form session expired';
                    $result['details']['expired'] = true;
                }
            }
        }

        // Extract email and subject for granular checks
        $email = $this->extract_email_from_data($data);
        $subject = $this->extract_subject_from_data($data);
        $body = $this->extract_body_from_data($data);
        $all_content = $this->flatten_content($data);

        // 6. Email Blacklist Check
        if ($email && $this->is_email_blacklisted($email)) {
            $result['score'] += 100;
            $result['reasons'][] = 'Email address blacklisted';
            $result['details']['email_blacklisted'] = true;
            $this->increment_blacklist_hits('email', strtolower($email));
        }

        // 7. Domain Blacklist Check (from email and content)
        $domains = $this->extract_domains($all_content);
        if ($email) {
            $email_domain = substr($email, strpos($email, '@') + 1);
            $domains[] = strtolower($email_domain);
        }
        $domains = array_unique($domains);

        foreach ($domains as $domain) {
            if ($this->is_domain_blacklisted($domain)) {
                $result['score'] += 90;
                $result['reasons'][] = "Domain blacklisted: {$domain}";
                $result['details']['domain_blacklisted'] = $domain;
                $this->increment_blacklist_hits('domain', $domain);
                break;
            }
        }

        // 8. Subject Keyword Check
        if ($subject && !empty($this->options['blacklist_keywords_subject'])) {
            $subject_keywords = $this->parse_blacklist($this->options['blacklist_keywords_subject']);
            foreach ($subject_keywords as $keyword) {
                if (stripos($subject, $keyword) !== false) {
                    $result['score'] += 90;
                    $result['reasons'][] = "Blocked keyword in subject: {$keyword}";
                    $result['details']['subject_keyword'] = $keyword;
                    break;
                }
            }
        }

        // 9. Body Keyword Check
        if ($body && !empty($this->options['blacklist_keywords_body'])) {
            $body_keywords = $this->parse_blacklist($this->options['blacklist_keywords_body']);
            foreach ($body_keywords as $keyword) {
                if (stripos($body, $keyword) !== false) {
                    $result['score'] += 85;
                    $result['reasons'][] = "Blocked keyword in body: {$keyword}";
                    $result['details']['body_keyword'] = $keyword;
                    break;
                }
            }
        }

        // 10. Global Keyword Check
        if (!empty($this->options['blacklist_keywords_all'])) {
            $global_keywords = $this->parse_blacklist($this->options['blacklist_keywords_all']);
            foreach ($global_keywords as $keyword) {
                if (stripos($all_content, $keyword) !== false) {
                    $result['score'] += 90;
                    $result['reasons'][] = "Blocked keyword: {$keyword}";
                    $result['details']['global_keyword'] = $keyword;
                    break;
                }
            }
        }

        // 11. Character Filtering
        if ($this->options['cyrillic_filter'] && preg_match('/[\p{Cyrillic}]/u', $all_content)) {
            $result['score'] += 60;
            $result['reasons'][] = 'Contains Cyrillic characters';
            $result['details']['cyrillic'] = true;
        }

        if ($this->options['hanzi_filter'] && preg_match('/[\p{Han}]/u', $all_content)) {
            $result['score'] += 60;
            $result['reasons'][] = 'Contains Chinese characters';
            $result['details']['hanzi'] = true;
        }

        // Bengali and South Asian scripts (Bengali, Devanagari, Tamil, Telugu, etc.)
        if ($this->options['bengali_filter'] && preg_match('/[\p{Bengali}\p{Devanagari}\p{Tamil}\p{Telugu}\p{Kannada}\p{Malayalam}\p{Gujarati}\p{Oriya}\p{Gurmukhi}]/u', $all_content)) {
            $result['score'] += 60;
            $result['reasons'][] = 'Contains South Asian script characters';
            $result['details']['bengali'] = true;
        }

        // Latin characters filter (blocks English/European languages)
        if ($this->options['latin_filter'] && preg_match('/[\p{Latin}]/u', $all_content)) {
            $result['score'] += 60;
            $result['reasons'][] = 'Contains Latin characters';
            $result['details']['latin'] = true;
        }

        // 12. Database Blacklist Check
        $db_blacklist_result = $this->check_database_blacklist($email, $domains, $all_content);
        if ($db_blacklist_result['matched']) {
            $result['score'] += $db_blacklist_result['score'];
            $result['reasons'] = array_merge($result['reasons'], $db_blacklist_result['reasons']);
            $result['details'] = array_merge($result['details'], $db_blacklist_result['details']);
        }

        // 13. Heuristic Analysis
        if ($this->options['heuristic_enabled']) {
            $heuristic = $this->heuristic_analysis($all_content);
            $result['score'] += $heuristic['score'];
            $result['reasons'] = array_merge($result['reasons'], $heuristic['reasons']);
            $result['details'] = array_merge($result['details'], $heuristic['details']);
        }

        // 14. Local Gibberish Detection (always runs for obvious keyboard mashing)
        $local_gibberish = $this->check_obvious_gibberish($data);
        if ($local_gibberish['score'] > 0) {
            $result['score'] += $local_gibberish['score'];
            $result['reasons'] = array_merge($result['reasons'], $local_gibberish['reasons']);
            $result['details'] = array_merge($result['details'], $local_gibberish['details']);
        }

        // 15. AI Gibberish Detection (only if enabled and not already spam)
        if ($result['score'] < $this->options['spam_threshold'] && $this->options['gibberish_detection_enabled']) {
            $gibberish = $this->check_gibberish($data);
            $result['score'] += $gibberish['score'];
            $result['reasons'] = array_merge($result['reasons'], $gibberish['reasons']);
            $result['details'] = array_merge($result['details'], $gibberish['details']);
        }

        // Determine spam status
        $result['is_spam'] = $result['score'] >= $this->options['spam_threshold'];

        // Log submission
        if ($result['is_spam'] || $this->options['log_clean_submissions']) {
            $this->log_submission($result, $data, $source, $email, $subject);
        }

        // Update rate limit counter
        if ($this->options['rate_limit_enabled']) {
            $this->increment_rate_limit($ip);
        }

        return $result;
    }

    /**
     * Heuristic analysis
     */
    private function heuristic_analysis($content) {
        $result = [
            'score' => 0,
            'reasons' => [],
            'details' => []
        ];

        // Link count
        preg_match_all('/https?:\/\/[^\s<>"\']+/i', $content, $links);
        $link_count = count($links[0]);
        if ($link_count > $this->options['max_links']) {
            $result['score'] += min(50, ($link_count - $this->options['max_links']) * 10);
            $result['reasons'][] = "Too many links ({$link_count})";
            $result['details']['link_count'] = $link_count;
        }

        // Spam patterns with scores
        $patterns = [
            '/\b(viagra|cialis|levitra|pharmacy|pills)\b/i' => ['score' => 40, 'name' => 'Pharma spam'],
            '/\b(casino|poker|gambling|bet365|slots)\b/i' => ['score' => 35, 'name' => 'Gambling spam'],
            '/\b(lottery|winner|jackpot|prize|congratulations.*won)\b/i' => ['score' => 35, 'name' => 'Lottery scam'],
            '/\b(crypto|bitcoin|ethereum|investment.*opportunity)\b/i' => ['score' => 30, 'name' => 'Crypto spam'],
            '/\b(make.*money.*fast|work.*from.*home|passive.*income)\b/i' => ['score' => 30, 'name' => 'MLM spam'],
            '/\b(SEO.*service|backlink|rank.*google|first.*page)\b/i' => ['score' => 40, 'name' => 'SEO spam'],
            '/\b(web.*design.*service|website.*development.*offer)\b/i' => ['score' => 25, 'name' => 'Service spam'],
            '/\b(dear.*sir|dear.*madam|to.*whom.*it.*may.*concern)\b/i' => ['score' => 20, 'name' => 'Generic greeting'],
            '/\b(nigerian.*prince|inheritance|beneficiary|overseas.*fund)\b/i' => ['score' => 50, 'name' => '419 scam'],
            '/(.)\1{5,}/' => ['score' => 15, 'name' => 'Repeated characters'],
            '/[A-Z\s]{20,}/' => ['score' => 15, 'name' => 'Excessive caps'],
            '/<[^>]*(script|iframe|object|embed|form)/i' => ['score' => 60, 'name' => 'HTML injection'],
            '/javascript:/i' => ['score' => 60, 'name' => 'JS injection'],
            '/\bon\w+\s*=/i' => ['score' => 50, 'name' => 'Event handler injection'],
        ];

        foreach ($patterns as $pattern => $config) {
            if (preg_match($pattern, $content)) {
                $result['score'] += $config['score'];
                $result['reasons'][] = $config['name'] . ' detected';
                $result['details']['pattern_' . sanitize_key($config['name'])] = true;
            }
        }

        // Multiple email addresses
        preg_match_all('/[\w.+-]+@[\w.-]+\.[a-z]{2,}/i', $content, $emails);
        if (count($emails[0]) > 3) {
            $result['score'] += 25;
            $result['reasons'][] = 'Multiple email addresses';
            $result['details']['email_count'] = count($emails[0]);
        }

        // Suspicious TLDs in links
        $suspicious_tlds = ['.ru', '.cn', '.tk', '.ml', '.ga', '.cf', '.gq', '.xyz', '.top', '.work', '.click'];
        foreach ($suspicious_tlds as $tld) {
            if (stripos($content, $tld) !== false) {
                $result['score'] += 15;
                $result['reasons'][] = "Suspicious TLD: {$tld}";
                $result['details']['suspicious_tld'] = $tld;
                break;
            }
        }

        return $result;
    }

    /**
     * =========================================================================
     * GIBBERISH DETECTION
     * =========================================================================
     */

    /**
     * Check for obvious gibberish/keyboard mashing (always runs, no settings required)
     * This catches patterns like "safasafafafaffff", "asdfasdfasdf", "qwertyqwerty"
     */
    private function check_obvious_gibberish($data) {
        $result = [
            'score' => 0,
            'reasons' => [],
            'details' => []
        ];

        // Fields to check for gibberish (common form field names)
        $check_fields = [
            'name', 'first', 'last', 'your-name', 'your-subject', 'subject', 'message', 'comment',
            'company', 'organization', 'title', 'city', 'address',
            // Lithuanian
            'vardas', 'pavarde', 'pavadinimas', 'zinute', 'pranesimas', 'komentaras', 'imone', 'miestas',
            // German
            'vorname', 'nachname', 'betreff', 'nachricht', 'firma',
            // French
            'nom', 'prenom', 'sujet', 'societe',
            // Spanish
            'nombre', 'apellido', 'asunto', 'mensaje', 'empresa'
        ];

        foreach ($data as $key => $value) {
            // Skip internal fields
            if (strpos($key, '_oh_') === 0 || strpos($key, '_wp') === 0 || strpos($key, '_wpcf7') === 0) {
                continue;
            }

            // Skip email and phone fields
            if (preg_match('/email|phone|tel|mail|fax/i', $key)) {
                continue;
            }

            // Check if this field should be examined
            $should_check = false;
            foreach ($check_fields as $field) {
                if (stripos($key, $field) !== false) {
                    $should_check = true;
                    break;
                }
            }

            if (!$should_check) {
                continue;
            }

            $text = is_array($value) ? implode(' ', $value) : strval($value);
            $text = trim($text);

            // Skip if too short or looks like valid data
            if (strlen($text) < 6) continue;
            if (filter_var($text, FILTER_VALIDATE_EMAIL)) continue;
            if (filter_var($text, FILTER_VALIDATE_URL)) continue;
            if (preg_match('/^[\d\s\+\-\(\)]+$/', $text)) continue;

            // Check for obvious gibberish patterns
            if ($this->is_obvious_gibberish($text)) {
                $result['score'] = 85; // High score for obvious gibberish
                $result['reasons'][] = 'Gibberish/random text in: ' . sanitize_text_field($key);
                $result['details']['gibberish_field'] = $key;
                $result['details']['gibberish_value'] = substr($text, 0, 50);
                break; // One gibberish field is enough to flag
            }
        }

        return $result;
    }

    /**
     * Check if text is obvious gibberish (keyboard mashing, random chars)
     */
    private function is_obvious_gibberish($text) {
        $condensed = preg_replace('/\s+/', '', $text);
        $len = mb_strlen($condensed, 'UTF-8');

        if ($len < 6) {
            return false;
        }

        // Check 1: Very low character variety (keyboard mashing like "safasafafafaffff")
        // Use mb_str_split for UTF-8 safety
        if (function_exists('mb_str_split')) {
            $chars = mb_str_split(mb_strtolower($condensed, 'UTF-8'), 1, 'UTF-8');
        } else {
            $chars = preg_split('//u', mb_strtolower($condensed, 'UTF-8'), -1, PREG_SPLIT_NO_EMPTY);
        }
        $unique_chars = count(array_unique($chars));
        $variety_ratio = $unique_chars / $len;

        // If text is 8+ chars with only 3-4 unique characters, it's likely gibberish
        if ($len >= 8 && $variety_ratio < 0.35) {
            return true;
        }

        // Check 2: Repeated short patterns (like "asdfasdf", "qwer qwer")
        $lower = mb_strtolower($condensed, 'UTF-8');
        if ($len >= 8) {
            // Check if text is made of repeating 2-4 char patterns
            for ($pattern_len = 2; $pattern_len <= 4; $pattern_len++) {
                $pattern = mb_substr($lower, 0, $pattern_len, 'UTF-8');
                $repeated = str_repeat($pattern, (int)ceil($len / $pattern_len));
                if (mb_substr($repeated, 0, $len, 'UTF-8') === $lower) {
                    return true;
                }
            }
        }

        // Check 3: Very low vowel ratio (almost no vowels)
        $vowels = preg_match_all('/[aeiouąęėįųūôöüäàáâãèéêëìíîïòóõøùúûÿæœyAEIOUÀÁÂÃÄÅÈÉÊËÌÍÎÏÒÓÔÕÖÙÚÛÜÝY]/u', $condensed);
        $vowel_ratio = $vowels / max(1, $len);
        if ($vowel_ratio < 0.08 && $len > 6) {
            return true;
        }

        // Check 4: Long consonant sequences (5+ consonants in a row)
        if (preg_match('/[bcdfghjklmnpqrstvwxzBCDFGHJKLMNPQRSTVWXZ]{5,}/', $condensed)) {
            return true;
        }

        // Check 5: Random mixed case pattern (like "QGNItWfBJmqYrUNcxjYOmdZ")
        $uppercase = preg_match_all('/[A-Z]/', $condensed);
        $lowercase = preg_match_all('/[a-z]/', $condensed);
        if ($uppercase > 3 && $lowercase > 3 && $len > 10) {
            $mid_upper = preg_match_all('/(?<=[a-z])[A-Z]/', $condensed);
            if ($mid_upper > 3) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check fields for gibberish/nonsensical text (AI-assisted)
     * Works with local pattern detection, optionally verified by AI
     */
    private function check_gibberish($data) {
        $result = [
            'score' => 0,
            'reasons' => [],
            'details' => []
        ];

        // Check if enabled (local detection works without API key)
        if (!$this->options['gibberish_detection_enabled']) {
            return $result;
        }

        $fields_to_check = $this->options['gibberish_fields'];
        $min_length = intval($this->options['gibberish_min_length']);
        $language = $this->options['gibberish_language'];

        $suspicious_fields = [];
        $obvious_gibberish = [];

        // Collect fields to check - scan ALL text fields for obvious gibberish
        foreach ($data as $key => $value) {
            // Skip internal fields
            if (strpos($key, '_oh_') === 0 || strpos($key, '_wp') === 0 || strpos($key, '_wpcf7') === 0) {
                continue;
            }

            // Skip email and phone fields by name
            if (preg_match('/email|phone|tel|mail/i', $key)) {
                continue;
            }

            $text = is_array($value) ? implode(' ', $value) : strval($value);
            $text = trim($text);

            // Skip if too short
            if (strlen($text) < $min_length) {
                continue;
            }

            // Skip if looks like email or phone or URL
            if (filter_var($text, FILTER_VALIDATE_EMAIL) ||
                preg_match('/^[\d\s\+\-\(\)]+$/', $text) ||
                filter_var($text, FILTER_VALIDATE_URL)) {
                continue;
            }

            // Check for obvious gibberish patterns (block immediately)
            $gibberish_result = $this->local_gibberish_check($text, true);
            if ($gibberish_result === 'obvious') {
                $obvious_gibberish[$key] = $text;
            } elseif ($gibberish_result === 'suspicious') {
                // Check if field name matches any target field pattern for AI verification
                foreach ($fields_to_check as $field_pattern) {
                    if (stripos($key, $field_pattern) !== false) {
                        $suspicious_fields[$key] = $text;
                        break;
                    }
                }
            }
        }

        // Obvious gibberish detected - block immediately without AI
        if (!empty($obvious_gibberish)) {
            $result['score'] = intval($this->options['gibberish_score']);
            $result['reasons'][] = 'Random/gibberish text detected in: ' . implode(', ', array_keys($obvious_gibberish));
            $result['details']['gibberish_fields'] = $obvious_gibberish;
            $result['details']['gibberish_detected'] = true;
            return $result;
        }

        // If we have suspicious fields and AI is configured, verify with AI
        if (!empty($suspicious_fields) && !empty($this->options['gibberish_api_key'])) {
            $ai_result = $this->verify_gibberish_with_ai($suspicious_fields, $language);

            if ($ai_result['is_gibberish']) {
                $result['score'] = intval($this->options['gibberish_score']);
                $result['reasons'][] = 'Gibberish/random text detected in: ' . implode(', ', array_keys($ai_result['fields']));
                $result['details']['gibberish_fields'] = $ai_result['fields'];
                $result['details']['gibberish_detected'] = true;
            }
        }

        return $result;
    }

    /**
     * Quick local check for obvious gibberish patterns
     * WPML-aware: balances detection with avoiding false positives on multilingual sites
     *
     * @param string $text Text to check
     * @param bool $return_level If true, returns 'obvious', 'suspicious', or false
     * @return bool|string Returns true/false or level string
     */
    private function local_gibberish_check($text, $return_level = false) {
        // Remove spaces for analysis
        $condensed = preg_replace('/\s+/', '', $text);
        $len = mb_strlen($condensed, 'UTF-8');

        if ($len < 5) {
            return false;
        }

        // ============ OBVIOUS GIBBERISH - block immediately ============

        // Very low character variety (same letters repeated - "aaaaaa", "abababab", "afsfasfsafsaf")
        // This catches keyboard mashing with limited keys
        $unique_chars = count(array_unique(mb_str_split(mb_strtolower($condensed, 'UTF-8'), 1, 'UTF-8')));
        $variety_ratio = $unique_chars / $len;
        if ($variety_ratio < 0.35 && $len >= 8) {
            return $return_level ? 'obvious' : true;
        }

        // Very low vowel ratio (almost no vowels)
        $vowels = preg_match_all('/[aeiouąęėįųūôöüäàáâãèéêëìíîïòóõøùúûÿæœyAEIOUÀÁÂÃÄÅÈÉÊËÌÍÎÏÒÓÔÕÖÙÚÛÜÝY]/u', $condensed);
        $vowel_ratio = $vowels / max(1, $len);
        if ($vowel_ratio < 0.1 && $len > 6) {
            return $return_level ? 'obvious' : true;
        }

        // Long consonant sequences (5+ consonants in a row)
        if (preg_match('/[bcdfghjklmnpqrstvwxzBCDFGHJKLMNPQRSTVWXZ]{5,}/', $condensed)) {
            return $return_level ? 'obvious' : true;
        }

        // Exact repetition pattern (first half = second half)
        $lower = mb_strtolower($condensed, 'UTF-8');
        if ($len >= 6) {
            $half = floor($len / 2);
            $first_half = mb_substr($lower, 0, $half, 'UTF-8');
            $second_half = mb_substr($lower, $half, $half, 'UTF-8');
            if ($first_half === $second_half && $half >= 3) {
                return $return_level ? 'obvious' : true;
            }
        }

        // Random mixed case in middle of text (random generators)
        $uppercase = preg_match_all('/[A-Z]/', $condensed);
        $lowercase = preg_match_all('/[a-z]/', $condensed);
        if ($uppercase > 3 && $lowercase > 3 && $len > 10) {
            $upper_ratio = $uppercase / ($uppercase + $lowercase);
            if ($upper_ratio > 0.35 && $upper_ratio < 0.65) {
                $mid_upper = preg_match_all('/(?<=[a-z])[A-Z]/', $condensed);
                if ($mid_upper > 3) {
                    return $return_level ? 'obvious' : true;
                }
            }
        }

        // ============ SUSPICIOUS - may need AI verification ============

        // Moderate consonant clusters (4 consonants in a row)
        $consonant_clusters = preg_match_all('/[bcdfghjklmnpqrstvwxzBCDFGHJKLMNPQRSTVWXZ]{4,}/i', $condensed);
        if ($consonant_clusters >= 1 && $len > 5) {
            return $return_level ? 'suspicious' : true;
        }

        // Low vowel ratio (but not extreme)
        if ($vowel_ratio < 0.15 && $len > 6) {
            return $return_level ? 'suspicious' : true;
        }

        // Low character variety (but not extreme)
        if ($variety_ratio < 0.4 && $len >= 10) {
            return $return_level ? 'suspicious' : true;
        }

        return false;
    }

    /**
     * Verify suspicious text with AI API
     */
    private function verify_gibberish_with_ai($fields, $language) {
        $result = [
            'is_gibberish' => false,
            'fields' => []
        ];

        $api_key = $this->options['gibberish_api_key'];
        $provider = $this->options['gibberish_ai_provider'];

        // Build the prompt
        $fields_text = '';
        foreach ($fields as $field_name => $value) {
            $fields_text .= "- {$field_name}: \"{$value}\"\n";
        }

        $language_names = [
            'en' => 'English',
            'lt' => 'Lithuanian',
            'lv' => 'Latvian',
            'et' => 'Estonian',
            'es' => 'Spanish',
            'de' => 'German',
            'fr' => 'French',
            'it' => 'Italian',
            'pt' => 'Portuguese',
            'nl' => 'Dutch',
            'pl' => 'Polish',
            'cs' => 'Czech',
            'sk' => 'Slovak',
            'hu' => 'Hungarian',
            'ro' => 'Romanian',
            'bg' => 'Bulgarian',
            'hr' => 'Croatian',
            'sl' => 'Slovenian',
            'uk' => 'Ukrainian',
            'ru' => 'Russian',
            'sv' => 'Swedish',
            'no' => 'Norwegian',
            'da' => 'Danish',
            'fi' => 'Finnish',
            'ja' => 'Japanese',
            'zh' => 'Chinese',
            'ko' => 'Korean',
            'auto' => 'any language',
        ];

        // Check for domain-specific languages first (supports multiple: "lt,en")
        $domain_langs = $this->get_domain_specific_languages();
        if ($domain_langs && is_array($domain_langs)) {
            // Convert language codes to names
            $lang_names_list = [];
            foreach ($domain_langs as $code) {
                if (isset($language_names[$code])) {
                    $lang_names_list[] = $language_names[$code];
                }
            }
            if (!empty($lang_names_list)) {
                if (count($lang_names_list) === 1) {
                    $lang_name = $lang_names_list[0];
                } else {
                    // Multiple languages: "Lithuanian or English"
                    $last = array_pop($lang_names_list);
                    $lang_name = implode(', ', $lang_names_list) . ' or ' . $last;
                }
            } else {
                $lang_name = 'any language';
            }
        }
        // WPML integration: detect current language if set to auto
        elseif ($language === 'auto' || empty($language)) {
            $wpml_lang = $this->get_wpml_current_language();
            if ($wpml_lang && isset($language_names[$wpml_lang])) {
                $lang_name = $language_names[$wpml_lang];
            } else {
                $lang_name = 'any language'; // Let AI detect
            }
        } else {
            $lang_name = $language_names[$language] ?? 'any language';
        }

        $prompt = "You are a spam detection assistant. Analyze these form field values and determine if they contain gibberish, random characters, or nonsensical text that a human would not type. The expected language is {$lang_name}.

Form fields to analyze:
{$fields_text}

For each field, respond with ONLY a JSON object in this exact format:
{\"gibberish_detected\": true/false, \"fields\": {\"field_name\": true/false, ...}}

Rules:
- Random letter combinations like 'QGNItWfBJmqY' are gibberish
- Real names, words, or phrases in any language are NOT gibberish
- Single words or short text that could be a name are NOT gibberish
- URLs, emails, phone numbers should be ignored
- Return ONLY the JSON, no explanation";

        try {
            if ($provider === 'openai') {
                $response = $this->call_openai_api($api_key, $prompt);
            } elseif ($provider === 'anthropic') {
                $response = $this->call_anthropic_api($api_key, $prompt);
            } else {
                return $result;
            }

            if ($response && isset($response['gibberish_detected'])) {
                $result['is_gibberish'] = (bool) $response['gibberish_detected'];
                if (isset($response['fields'])) {
                    foreach ($response['fields'] as $field => $is_gibberish) {
                        if ($is_gibberish) {
                            $result['fields'][$field] = $fields[$field] ?? '';
                        }
                    }
                }
            }
        } catch (Exception $e) {
            // Log error but don't block submission on API failure
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG enabled
                error_log('HoneypotGuard AI Error: ' . $e->getMessage());
            }
        }

        return $result;
    }

    /**
     * Call OpenAI API
     */
    private function call_openai_api($api_key, $prompt) {
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
            'timeout' => 10,
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode([
                'model' => 'gpt-4o-mini',
                'messages' => [
                    ['role' => 'user', 'content' => $prompt]
                ],
                'max_tokens' => 200,
                'temperature' => 0.1,
            ]),
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['error'])) {
            throw new Exception(esc_html($body['error']['message'] ?? 'OpenAI API error'));
        }

        if (isset($body['choices'][0]['message']['content'])) {
            $content = $body['choices'][0]['message']['content'];
            // Extract JSON from response
            if (preg_match('/\{.*\}/s', $content, $matches)) {
                return json_decode($matches[0], true);
            }
        }

        return null;
    }

    /**
     * Call Anthropic API
     */
    private function call_anthropic_api($api_key, $prompt) {
        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'timeout' => 10,
            'headers' => [
                'x-api-key' => $api_key,
                'Content-Type' => 'application/json',
                'anthropic-version' => '2023-06-01',
            ],
            'body' => json_encode([
                'model' => 'claude-3-haiku-20240307',
                'max_tokens' => 200,
                'messages' => [
                    ['role' => 'user', 'content' => $prompt]
                ],
            ]),
        ]);

        if (is_wp_error($response)) {
            throw new Exception(esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['error'])) {
            throw new Exception(esc_html($body['error']['message'] ?? 'Anthropic API error'));
        }

        if (isset($body['content'][0]['text'])) {
            $content = $body['content'][0]['text'];
            // Extract JSON from response
            if (preg_match('/\{.*\}/s', $content, $matches)) {
                return json_decode($matches[0], true);
            }
        }

        return null;
    }

    /**
     * =========================================================================
     * BLACKLIST HELPERS
     * =========================================================================
     */

    /**
     * Parse blacklist string to array
     */
    private function parse_blacklist($string) {
        if (empty($string)) return [];
        return array_filter(array_map('trim', explode("\n", strtolower($string))));
    }

    /**
     * Check if IP is blacklisted
     */
    private function is_ip_blacklisted($ip) {
        // Check options blacklist
        $blacklist = $this->parse_blacklist($this->options['blacklist_ips']);
        foreach ($blacklist as $blocked_ip) {
            if ($this->ip_matches($ip, $blocked_ip)) {
                return true;
            }
        }

        // Check database blacklist
        return $this->is_in_database_blacklist('ip', $ip);
    }

    /**
     * Check IP match (supports CIDR notation)
     */
    private function ip_matches($ip, $pattern) {
        if (strpos($pattern, '/') !== false) {
            list($subnet, $mask) = explode('/', $pattern);
            $ip_long = ip2long($ip);
            $subnet_long = ip2long($subnet);
            $mask_long = ~((1 << (32 - $mask)) - 1);
            return ($ip_long & $mask_long) === ($subnet_long & $mask_long);
        }
        return $ip === $pattern;
    }

    /**
     * Check if email is blacklisted
     */
    private function is_email_blacklisted($email) {
        $email = strtolower($email);

        // Check options blacklist
        $blacklist = $this->parse_blacklist($this->options['blacklist_emails']);
        foreach ($blacklist as $blocked) {
            if ($email === $blocked || fnmatch($blocked, $email)) {
                return true;
            }
        }

        return $this->is_in_database_blacklist('email', $email);
    }

    /**
     * Check if domain is blacklisted
     */
    private function is_domain_blacklisted($domain) {
        $domain = strtolower($domain);

        // Check options blacklist
        $blacklist = $this->parse_blacklist($this->options['blacklist_domains']);
        foreach ($blacklist as $blocked) {
            if ($domain === $blocked || $this->domain_matches($domain, $blocked)) {
                return true;
            }
        }

        return $this->is_in_database_blacklist('domain', $domain);
    }

    /**
     * Check domain match (supports wildcards)
     */
    private function domain_matches($domain, $pattern) {
        // Match subdomains
        if (strpos($pattern, '*.') === 0) {
            $base = substr($pattern, 2);
            return $domain === $base || substr($domain, -strlen($base) - 1) === '.' . $base;
        }
        return $domain === $pattern;
    }

    /**
     * Check database blacklist
     */
    private function is_in_database_blacklist($type, $value) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table lookup
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->blacklist_table} WHERE type = %s AND value = %s",
            $type,
            strtolower($value)
        ));
        return $count > 0;
    }

    /**
     * Check all database blacklists (with caching)
     */
    private function check_database_blacklist($email, $domains, $content) {
        $result = [
            'matched' => false,
            'score' => 0,
            'reasons' => [],
            'details' => []
        ];

        // Get cached blacklist entries
        $entries = $this->get_cached_blacklist();

        if (empty($entries)) {
            return $result;
        }

        $ip = $this->get_client_ip();

        foreach ($entries as $entry) {
            $matched = false;

            switch ($entry->type) {
                case 'email':
                    if ($email && strtolower($email) === $entry->value) {
                        $matched = true;
                    }
                    break;
                case 'domain':
                    foreach ($domains as $domain) {
                        if ($this->domain_matches($domain, $entry->value)) {
                            $matched = true;
                            break;
                        }
                    }
                    break;
                case 'keyword':
                    if (stripos($content, $entry->value) !== false) {
                        $matched = true;
                    }
                    break;
                case 'ip':
                    if ($this->ip_matches($ip, $entry->value)) {
                        $matched = true;
                    }
                    break;
            }

            if ($matched) {
                $result['matched'] = true;
                $result['score'] += 100;
                $result['reasons'][] = "Database blacklist match: {$entry->type}";
                $result['details']['db_blacklist_' . $entry->type] = $entry->value;
                $this->increment_blacklist_hits($entry->type, $entry->value);
            }
        }

        return $result;
    }

    /**
     * Get cached blacklist entries
     */
    private function get_cached_blacklist() {
        // Return from memory cache if available
        if ($this->cached_blacklist !== null) {
            return $this->cached_blacklist;
        }

        // Try object cache
        $cache_key = 'honeypot_guard_blacklist_entries';
        $entries = wp_cache_get($cache_key, 'honeypot_guard');

        if ($entries === false) {
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, results cached
            $entries = $wpdb->get_results("SELECT type, value FROM {$this->blacklist_table}");

            // Cache for 5 minutes
            wp_cache_set($cache_key, $entries, 'honeypot_guard', 300);
        }

        $this->cached_blacklist = $entries;
        return $entries;
    }

    /**
     * Clear blacklist cache (called when blacklist is modified)
     */
    public function clear_blacklist_cache() {
        wp_cache_delete('honeypot_guard_blacklist_entries', 'honeypot_guard');
        $this->cached_blacklist = null;
    }

    /**
     * Increment blacklist hit counter
     */
    private function increment_blacklist_hits($type, $value) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table update
        $wpdb->query($wpdb->prepare(
            "UPDATE {$this->blacklist_table} SET hits = hits + 1 WHERE type = %s AND value = %s",
            $type,
            strtolower($value)
        ));
    }

    /**
     * =========================================================================
     * RATE LIMITING
     * =========================================================================
     */

    /**
     * Check if IP is rate limited
     */
    private function is_rate_limited($ip) {
        $key = $this->rate_limit_transient . md5($ip);
        $count = get_transient($key);
        return $count !== false && $count >= $this->options['rate_limit_submissions'];
    }

    /**
     * Increment rate limit counter
     */
    private function increment_rate_limit($ip) {
        $key = $this->rate_limit_transient . md5($ip);
        $count = get_transient($key);

        if ($count === false) {
            set_transient($key, 1, $this->options['rate_limit_window']);
        } else {
            set_transient($key, $count + 1, $this->options['rate_limit_window']);
        }
    }

    /**
     * Add rate-limited IP to blacklist
     */
    private function add_rate_limited_ip_to_blacklist($ip) {
        global $wpdb;
        $table = $wpdb->prefix . 'honeypot_guard_blacklist';

        // Check if IP already exists in blacklist
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table lookup
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE type = 'ip' AND value = %s",
            $ip
        ));

        if ($existing) {
            // Update hits count
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table update
            $wpdb->query($wpdb->prepare(
                "UPDATE {$table} SET hits = hits + 1 WHERE id = %d",
                $existing
            ));
        } else {
            // Add new entry
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert
            $wpdb->insert($table, [
                'type' => 'ip',
                'value' => $ip,
                'reason' => __('Rate limit exceeded - auto-blocked', 'honeypot-guard-silent-anti-spam'),
                'hits' => 1,
                'created_at' => current_time('mysql')
            ], ['%s', '%s', '%s', '%d', '%s']);
        }
    }

    /**
     * =========================================================================
     * CONTENT EXTRACTION
     * =========================================================================
     */

    /**
     * Extract email from form data
     */
    private function extract_email_from_data($data) {
        $email_fields = ['email', 'your-email', 'user_email', 'contact_email', 'mail', 'e-mail', 'from'];
        foreach ($email_fields as $field) {
            if (!empty($data[$field]) && is_email($data[$field])) {
                return sanitize_email($data[$field]);
            }
        }
        // Search all fields for email pattern
        foreach ($data as $value) {
            if (is_string($value) && is_email($value)) {
                return sanitize_email($value);
            }
        }
        return null;
    }

    /**
     * Extract subject from form data
     */
    private function extract_subject_from_data($data) {
        $subject_fields = ['subject', 'your-subject', 'title', 'topic', 'inquiry', 'reason'];
        foreach ($subject_fields as $field) {
            if (!empty($data[$field])) {
                return sanitize_text_field($data[$field]);
            }
        }
        return null;
    }

    /**
     * Extract body/message from form data
     */
    private function extract_body_from_data($data) {
        $body_fields = ['message', 'your-message', 'body', 'content', 'comment', 'description', 'inquiry', 'textarea'];
        foreach ($body_fields as $field) {
            if (!empty($data[$field])) {
                return sanitize_textarea_field($data[$field]);
            }
        }
        return null;
    }

    /**
     * Extract domains from content
     */
    private function extract_domains($content) {
        $domains = [];

        // From URLs
        preg_match_all('/https?:\/\/([^\/\s<>"\']+)/i', $content, $url_matches);
        if (!empty($url_matches[1])) {
            foreach ($url_matches[1] as $domain) {
                $domain = strtolower(preg_replace('/^www\./', '', $domain));
                $domains[] = $domain;
            }
        }

        // From emails
        preg_match_all('/[\w.+-]+@([\w.-]+\.[a-z]{2,})/i', $content, $email_matches);
        if (!empty($email_matches[1])) {
            foreach ($email_matches[1] as $domain) {
                $domains[] = strtolower($domain);
            }
        }

        return array_unique($domains);
    }

    /**
     * Flatten form data to string
     */
    private function flatten_content($data) {
        $content = '';
        $exclude = ['_oh_timestamp', '_oh_nonce', '_wp_http_referer', 'action', '_wpnonce'];

        foreach ($data as $key => $value) {
            if (in_array($key, $exclude) || strpos($key, '_oh_') === 0) {
                continue;
            }
            if (is_array($value)) {
                $content .= ' ' . implode(' ', array_map('sanitize_text_field', $value));
            } else {
                $content .= ' ' . sanitize_text_field($value);
            }
        }

        return trim($content);
    }

    /**
     * =========================================================================
     * UTILITY METHODS
     * =========================================================================
     */

    /**
     * Check if current user is whitelisted
     */
    private function is_whitelisted_user() {
        if (!$this->options['whitelist_logged_in']) {
            return false;
        }

        if (!is_user_logged_in()) {
            return false;
        }

        $user = wp_get_current_user();
        $whitelisted_roles = $this->options['whitelist_roles'] ?? ['administrator', 'editor'];

        foreach ($whitelisted_roles as $role) {
            if (in_array($role, $user->roles)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get honeypot field name (with caching for performance)
     */
    public function get_honeypot_field_name() {
        // Return cached name if available (for same request)
        if ($this->cached_honeypot_name !== null) {
            return $this->cached_honeypot_name;
        }

        $mode = $this->options['honeypot_mode'] ?? 'auto';

        switch ($mode) {
            case 'custom':
                // Use user-specified field name
                $name = $this->options['honeypot_field_name'] ?? 'middle_name';
                break;

            case 'dynamic':
                // Dynamic rotation based on time setting
                $name = $this->get_dynamic_honeypot_name();
                break;

            case 'auto':
            default:
                // Auto-generate based on site hash (consistent per site)
                $hash = md5(get_site_url() . 'honeypot_guard' . wp_salt());
                $index = hexdec(substr($hash, 0, 2)) % count($this->honeypot_names);
                $name = $this->honeypot_names[$index];
                break;
        }

        $this->cached_honeypot_name = $name;
        return $name;
    }

    /**
     * Get dynamically rotated honeypot field name
     */
    private function get_dynamic_honeypot_name() {
        $rotation = $this->options['honeypot_rotation'] ?? 'session';
        $names = $this->get_honeypot_name_pool();

        switch ($rotation) {
            case 'hourly':
                // Changes every hour
                $seed = gmdate('YmdH') . get_site_url();
                break;

            case 'daily':
                // Changes every day
                $seed = gmdate('Ymd') . get_site_url();
                break;

            case 'session':
            default:
                // Changes per session (using session ID or visitor fingerprint)
                $seed = $this->get_visitor_fingerprint();
                break;
        }

        $hash = md5($seed . wp_salt());
        $index = hexdec(substr($hash, 0, 4)) % count($names);

        return $names[$index];
    }

    /**
     * Get the pool of honeypot names (built-in + custom)
     */
    private function get_honeypot_name_pool() {
        $names = $this->honeypot_names;

        // Add user's custom names if provided
        if (!empty($this->options['honeypot_custom_names'])) {
            $custom = array_filter(
                array_map('trim', explode("\n", $this->options['honeypot_custom_names']))
            );
            $names = array_merge($names, $custom);
        }

        return array_unique($names);
    }

    /**
     * Get visitor fingerprint for session-based rotation
     */
    private function get_visitor_fingerprint() {
        $parts = [
            $this->get_client_ip(),
            isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
            isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_ACCEPT_LANGUAGE'])) : '',
        ];

        return md5(implode('|', $parts));
    }

    /**
     * Get language(s) for current domain from domain-specific settings
     * Format: "domain.com:en" or "domain.com:lt,en" (multiple languages)
     * Returns array of language codes or null if not configured
     */
    private function get_domain_specific_languages() {
        $domain_languages = $this->options['gibberish_domain_languages'] ?? '';
        if (empty($domain_languages)) {
            return null;
        }

        // Get current domain
        $current_host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
        if (empty($current_host)) {
            return null;
        }

        // Remove www. prefix for matching
        $current_host = preg_replace('/^www\./i', '', strtolower($current_host));

        // Parse domain:language pairs
        $lines = array_filter(array_map('trim', explode("\n", $domain_languages)));
        foreach ($lines as $line) {
            // Skip comments
            if (strpos($line, '#') === 0) {
                continue;
            }

            // Parse domain:language format
            if (strpos($line, ':') !== false) {
                list($domain, $langs) = array_map('trim', explode(':', $line, 2));
                $domain = preg_replace('/^www\./i', '', strtolower($domain));

                // Exact match or subdomain match
                if ($current_host === $domain ||
                    substr($current_host, -strlen('.' . $domain)) === '.' . $domain) {
                    // Parse comma-separated languages (e.g., "lt,en")
                    $lang_array = array_filter(array_map('trim', explode(',', strtolower($langs))));
                    return !empty($lang_array) ? $lang_array : null;
                }
            }
        }

        return null;
    }

    /**
     * Get current WPML language code
     * Returns the current language if WPML is active, null otherwise
     */
    private function get_wpml_current_language() {
        // WPML
        if (defined('ICL_LANGUAGE_CODE')) {
            return ICL_LANGUAGE_CODE;
        }

        // WPML function
        if (function_exists('wpml_get_current_language')) {
            return wpml_get_current_language();
        }

        // Polylang
        if (function_exists('pll_current_language')) {
            return pll_current_language();
        }

        // TranslatePress - access via $GLOBALS to avoid unprefixed global statement
        if (!empty($GLOBALS['TRP_LANGUAGE'])) {
            return substr($GLOBALS['TRP_LANGUAGE'], 0, 2); // Get 2-letter code
        }

        // WordPress site language as fallback
        $locale = get_locale();
        if ($locale) {
            return substr($locale, 0, 2);
        }

        return null;
    }

    /**
     * Get multiple honeypot field names (for multi-honeypot mode)
     */
    public function get_honeypot_field_names($count = 2) {
        $names = $this->get_honeypot_name_pool();
        $seed = $this->get_visitor_fingerprint() . wp_salt();

        $selected = [];
        for ($i = 0; $i < $count && $i < count($names); $i++) {
            $hash = md5($seed . $i);
            $index = hexdec(substr($hash, 0, 4)) % count($names);

            // Avoid duplicates
            $attempts = 0;
            while (in_array($names[$index], $selected) && $attempts < 10) {
                $index = ($index + 1) % count($names);
                $attempts++;
            }

            $selected[] = $names[$index];
        }

        return $selected;
    }

    /**
     * Get client IP with proxy support
     */
    private function get_client_ip() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'HTTP_CLIENT_IP',
            'REMOTE_ADDR'
        ];

        foreach ($headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                $ip = sanitize_text_field(wp_unslash($_SERVER[$header]));
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }

        return '0.0.0.0';
    }

    /**
     * Log submission
     */
    private function log_submission($result, $data, $source, $email = null, $subject = null) {
        global $wpdb;

        // Ensure table exists
        $this->maybe_create_tables();

        // Prepare log data (remove internal fields)
        $log_data = is_array($data) ? $data : [];
        unset($log_data['_oh_timestamp'], $log_data['_oh_nonce'], $log_data['_wp_http_referer'], $log_data['_wpcf7'], $log_data['_wpcf7_version'], $log_data['_wpcf7_locale'], $log_data['_wpcf7_unit_tag'], $log_data['_wpcf7_container_post']);

        // Build reason string
        $reasons = isset($result['reasons']) && is_array($result['reasons'])
            ? array_slice($result['reasons'], 0, 10)
            : [];
        $reason_str = implode('; ', $reasons);

        // Get source URL (referer)
        $source_url = '';
        if (isset($_SERVER['HTTP_REFERER']) && !empty($_SERVER['HTTP_REFERER'])) {
            $source_url = esc_url_raw(substr(wp_unslash($_SERVER['HTTP_REFERER']), 0, 500));
        } elseif (isset($_SERVER['REQUEST_URI']) && !empty($_SERVER['REQUEST_URI'])) {
            $source_url = home_url(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])));
        }

        // Insert log entry
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert
        $inserted = $wpdb->insert(
            $this->table_name,
            [
                'status' => !empty($result['is_spam']) ? 'spam' : 'clean',
                'reason' => $reason_str ?: 'No issues detected',
                'content' => wp_json_encode($log_data),
                'ip_address' => $this->get_client_ip(),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT'])
                    ? sanitize_text_field(substr(wp_unslash($_SERVER['HTTP_USER_AGENT']), 0, 500))
                    : '',
                'form_source' => sanitize_text_field($source),
                'source_url' => $source_url,
                'spam_score' => min(255, max(0, intval($result['score'] ?? 0))),
                'email_field' => $email ? sanitize_email($email) : null,
                'subject_field' => $subject ? sanitize_text_field(substr($subject, 0, 255)) : null,
                'created_at' => current_time('mysql')
            ],
            ['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s']
        );

        // Log error if insert failed (for debugging)
        if ($inserted === false && defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG enabled
            error_log('Honeypot Guard: Failed to log submission - ' . $wpdb->last_error);
        }
    }

    /**
     * =========================================================================
     * FORM INTEGRATIONS
     * =========================================================================
     */

    /**
     * Contact Form 7 spam check
     * This runs on the wpcf7_spam filter which is the proper place to block spam.
     * Uses $submission->get_posted_data() for full access to all CF7 form fields.
     */
    public function check_cf7_spam($spam, $submission) {
        // If already marked spam by another plugin, don't interfere
        if ($spam) {
            return $spam;
        }

        try {
            // Get full submission data from CF7 (includes ALL form fields)
            $data = [];
            if ($submission && is_object($submission) && method_exists($submission, 'get_posted_data')) {
                $data = $submission->get_posted_data();
            }

            // Fallback to POST data if submission object doesn't work
            if (empty($data) && !empty($_POST)) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- CF7 handles nonce verification
                $data = $this->sanitize_submission_data($this->get_relevant_post_fields());
            }

            // If no data, can't check - allow through
            if (empty($data)) {
                return false;
            }

            // Check if whitelisted user
            if ($this->is_whitelisted_user()) {
                $this->cf7_spam_handled = true;
                $this->cf7_last_result = [
                    'is_spam' => false,
                    'score' => 0,
                    'reasons' => ['Whitelisted user'],
                    'details' => ['whitelisted' => true]
                ];
                return false;
            }

            // Sanitize CF7 posted data (CF7's get_posted_data returns unsanitized values)
            $data = $this->sanitize_submission_data($data);

            // Run full spam analysis (includes honeypot, blacklists, heuristics, timestamp)
            $result = $this->analyze_submission($data, 'contact-form-7');

            // Cache result for customize_cf7_ajax_response / customize_cf7_spam_message
            $this->cf7_last_result = $result;
            $this->cf7_spam_handled = true;

            // Return true to block spam, false to allow through
            return $result['is_spam'];

        } catch (Exception $e) {
            // Log error but don't block submission on error
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG enabled
                error_log('Honeypot Guard CF7 error: ' . $e->getMessage());
            }
            return false;
        }
    }

    /**
     * Log successful CF7 submissions
     * Note: Only logs if analyze_submission didn't already log (when log_clean_submissions is enabled)
     */
    public function log_cf7_success($contact_form) {
        // Skip if analyze_submission in check_cf7_spam already logged this
        // (analyze_submission logs spam always, and clean if log_clean_submissions is on)
        if ($this->cf7_spam_handled && $this->cf7_last_result !== null
            && $this->options['log_clean_submissions']) {
            return;
        }

        $submission = WPCF7_Submission::get_instance();
        if (!$submission) return;

        $data = $this->sanitize_submission_data($submission->get_posted_data());
        $email = $this->extract_email_from_data($data);
        $subject = $this->extract_subject_from_data($data);

        $result = [
            'is_spam' => false,
            'score' => isset($this->cf7_last_result['score']) ? $this->cf7_last_result['score'] : 0,
            'reasons' => ['Successful CF7 submission'],
            'details' => ['cf7_status' => 'mail_sent']
        ];

        $this->log_submission($result, $data, 'contact-form-7', $email, $subject);
    }

    /**
     * CF7 validation hook - check for spam at validation stage
     * This prevents form submission before it starts, which:
     * 1. Shows proper error message to users
     * 2. Prevents Meta Pixel/analytics from tracking spam as submissions
     */
    public function cf7_validate_log($result, $tags) {
        // Do NOT analyze spam here. The wpcf7_validate filter runs before
        // wpcf7_spam, and analyzing here causes problems:
        // 1. CF7's $submission->get_posted_data() is not yet available,
        //    so we'd use get_relevant_post_fields() which misses custom fields.
        // 2. analyze_submission() increments the rate limit counter, meaning
        //    each submission costs a rate-limit slot before spam check even runs.
        // 3. Pre-caching the result with incomplete data leads to wrong decisions.
        //
        // All analysis is now done in check_cf7_spam() which has access to the
        // full CF7 posted data via $submission->get_posted_data().
        return $result;
    }

    /**
     * Track if CF7 spam was already handled at validation stage
     */
    private $cf7_spam_handled = false;

    /**
     * Store last CF7 analysis result to avoid re-analyzing
     */
    private $cf7_last_result = null;

    /**
     * Customize CF7 spam message with proper localization
     */
    public function customize_cf7_spam_message($message, $status) {
        if ($status !== 'spam') {
            return $message;
        }

        // Only replace the message when OUR plugin flagged it as spam
        if ($this->cf7_spam_handled && $this->cf7_last_result !== null && !empty($this->cf7_last_result['is_spam'])) {
            return $this->get_translated_spam_message();
        }

        return $message;
    }

    /**
     * Customize CF7 AJAX response to ensure spam message is displayed
     */
    public function customize_cf7_ajax_response($response, $result) {
        // Only customize the response when OUR plugin flagged it as spam
        // Do not hijack spam responses from other plugins (Akismet, CF7 built-in, etc.)
        if (isset($response['status']) && $response['status'] === 'spam'
            && $this->cf7_spam_handled && $this->cf7_last_result !== null
            && !empty($this->cf7_last_result['is_spam'])
        ) {
            $response['status'] = 'aborted';
            $response['message'] = $this->get_translated_spam_message();
            if (!isset($response['invalid_fields'])) {
                $response['invalid_fields'] = [];
            }
        }

        // Log validation failures (missing required fields, invalid email, etc.)
        // This hook fires on every CF7 AJAX response, so it safely catches validation_failed
        if (isset($response['status']) && $response['status'] === 'validation_failed') {
            $this->log_cf7_validation_failure();
        }

        return $response;
    }

    /**
     * Log CF7 validation failures (missing/invalid required fields)
     */
    private function log_cf7_validation_failure() {
        $data = [];
        if (!empty($_POST)) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- CF7 handles nonce verification
            $data = $this->sanitize_submission_data($this->get_relevant_post_fields());
        }
        // Also try CF7 submission object for fuller data
        if (class_exists('WPCF7_Submission')) {
            $submission = WPCF7_Submission::get_instance();
            if ($submission && method_exists($submission, 'get_posted_data')) {
                $cf7_data = $submission->get_posted_data();
                if (!empty($cf7_data)) {
                    $data = $this->sanitize_submission_data($cf7_data);
                }
            }
        }
        if (empty($data)) {
            return;
        }

        $email = $this->extract_email_from_data($data);
        $subject = $this->extract_subject_from_data($data);

        $log_result = [
            'is_spam' => false,
            'score' => 0,
            'reasons' => ['CF7 validation failed (missing or invalid fields)'],
            'details' => ['cf7_status' => 'validation_failed']
        ];

        $this->log_submission($log_result, $data, 'contact-form-7', $email, $subject);
    }

    /**
     * Get translated spam message based on WPML/Polylang/domain settings
     */
    private function get_translated_spam_message() {
        $mode = $this->options['spam_message_mode'] ?? 'auto';

        // Mode: custom - use single custom message for all
        if ($mode === 'custom' && !empty($this->options['custom_spam_message'])) {
            return $this->options['custom_spam_message'];
        }

        // Always check per-domain messages first (works in any mode if configured)
        // This allows users to set domain-specific messages even in auto mode
        $domain_message = $this->get_domain_spam_message();
        if ($domain_message) {
            return $domain_message;
        }

        // Mode: per_domain - fall back to default message if set
        if ($mode === 'per_domain') {
            if (!empty($this->options['spam_message_default'])) {
                return $this->options['spam_message_default'];
            }
        }

        // Mode: auto - use language-based messages
        $lang = $this->get_current_language();

        // Also detect language from domain TLD if language is still 'en'
        if ($lang === 'en' || empty($lang)) {
            $current_host = isset($_SERVER['HTTP_HOST']) ? strtolower(sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']))) : '';
            $current_host = preg_replace('/^www\./', '', $current_host);

            // Map common TLDs to languages
            $tld_lang_map = [
                'lt' => 'lt', 'de' => 'de', 'fr' => 'fr', 'es' => 'es',
                'it' => 'it', 'pl' => 'pl', 'ru' => 'ru', 'nl' => 'nl',
                'pt' => 'pt', 'br' => 'pt', 'at' => 'de', 'ch' => 'de',
                'be' => 'nl', 'mx' => 'es', 'ar' => 'es', 'ua' => 'ru',
            ];

            // Extract TLD from domain
            $parts = explode('.', $current_host);
            $tld = end($parts);

            if (isset($tld_lang_map[$tld])) {
                $lang = $tld_lang_map[$tld];
            }
        }

        // Built-in spam messages in different languages
        $messages = [
            'en' => 'Your message could not be sent. Please verify your information and try again.',
            'lt' => 'Jūsų žinutė negalėjo būti išsiųsta. Patikrinkite savo informaciją ir bandykite dar kartą.',
            'de' => 'Ihre Nachricht konnte nicht gesendet werden. Bitte überprüfen Sie Ihre Angaben und versuchen Sie es erneut.',
            'fr' => 'Votre message n\'a pas pu être envoyé. Veuillez vérifier vos informations et réessayer.',
            'es' => 'Su mensaje no pudo ser enviado. Por favor verifique su información e intente nuevamente.',
            'it' => 'Il tuo messaggio non può essere inviato. Verifica le tue informazioni e riprova.',
            'pl' => 'Twoja wiadomość nie mogła zostać wysłana. Sprawdź swoje dane i spróbuj ponownie.',
            'ru' => 'Ваше сообщение не может быть отправлено. Пожалуйста, проверьте информацию и попробуйте снова.',
            'nl' => 'Uw bericht kon niet worden verzonden. Controleer uw gegevens en probeer het opnieuw.',
            'pt' => 'Sua mensagem não pôde ser enviada. Verifique suas informações e tente novamente.',
        ];

        // Apply filter to allow adding custom translations
        $messages = apply_filters('honeypot_guard_spam_messages', $messages);

        return $messages[$lang] ?? $messages['en'];
    }

    /**
     * Get spam message for current domain
     */
    private function get_domain_spam_message() {
        $per_domain = $this->options['spam_messages_per_domain'] ?? '';
        if (empty($per_domain)) {
            return null;
        }

        // Get current domain
        $current_host = isset($_SERVER['HTTP_HOST']) ? strtolower(sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST']))) : '';
        $current_host = preg_replace('/^www\./', '', $current_host); // Remove www prefix

        // Parse domain|message pairs
        $lines = explode("\n", $per_domain);
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line) || strpos($line, '|') === false) {
                continue;
            }

            list($domain, $message) = explode('|', $line, 2);
            $domain = strtolower(trim($domain));
            $domain = preg_replace('/^www\./', '', $domain); // Remove www prefix
            $message = trim($message);

            if ($domain === $current_host && !empty($message)) {
                return $message;
            }
        }

        return null;
    }

    /**
     * Get current language from WPML, Polylang, or domain settings
     */
    private function get_current_language() {
        // Check domain-specific language settings first
        $domain_langs = $this->get_domain_specific_languages();
        if (!empty($domain_langs)) {
            // Return first language for the domain
            return $domain_langs[0];
        }

        // WPML
        if (defined('ICL_LANGUAGE_CODE')) {
            return ICL_LANGUAGE_CODE;
        }

        // Polylang
        if (function_exists('pll_current_language')) {
            $lang = pll_current_language();
            if ($lang) {
                return $lang;
            }
        }

        // WordPress site language
        $locale = get_locale();
        return substr($locale, 0, 2);
    }

    /**
     * Add CF7 honeypot field
     */
    public function add_cf7_honeypot_field() {
        wpcf7_add_form_tag('honeypot-guard', [$this, 'cf7_honeypot_tag_handler']);
    }

    /**
     * CF7 honeypot tag handler
     */
    public function cf7_honeypot_tag_handler($tag) {
        $field_name = $this->get_honeypot_field_name();
        return sprintf(
            '<span class="oh-hp-wrap" style="position:absolute!important;left:-9999px!important;top:-9999px!important;visibility:hidden!important;"><label for="oh_%1$s">Leave empty</label><input type="text" name="%1$s" id="oh_%1$s" value="" tabindex="-1" autocomplete="new-password"></span>',
            esc_attr($field_name)
        );
    }

    /**
     * Add CF7 hidden fields
     */
    public function add_cf7_hidden_fields($fields) {
        $fields['_oh_timestamp'] = time();
        $fields['_oh_nonce'] = wp_create_nonce('honeypot_guard_form');
        return $fields;
    }

    /**
     * WPForms integration
     */
    public function check_wpforms_spam($form_data, $entry) {
        if (empty($entry['fields'])) {
            return $form_data;
        }

        $data = [];
        foreach ($entry['fields'] as $field) {
            $data[$field['name'] ?? $field['id']] = $field['value'] ?? '';
        }

        $result = $this->analyze_submission($data, 'wpforms');

        if ($result['is_spam']) {
            wpforms()->process->errors[$form_data['id']]['header'] = $this->get_spam_message();
        }

        return $form_data;
    }

    /**
     * Gravity Forms integration
     */
    public function check_gravity_forms_spam($is_spam, $form, $entry) {
        if ($is_spam) return $is_spam;

        $result = $this->analyze_submission($entry, 'gravity-forms');
        return $result['is_spam'];
    }

    /**
     * Ninja Forms integration
     */
    public function check_ninja_forms_spam($form_data) {
        if (empty($form_data['fields'])) {
            return $form_data;
        }

        $data = [];
        foreach ($form_data['fields'] as $field) {
            $key = $field['key'] ?? $field['id'] ?? 'field_' . count($data);
            $data[$key] = $field['value'] ?? '';
        }

        $result = $this->analyze_submission($data, 'ninja-forms');

        if ($result['is_spam']) {
            $form_data['errors']['form']['spam'] = $this->get_spam_message();
        }

        return $form_data;
    }

    /**
     * WordPress Comment spam check
     */
    public function check_comment_spam($commentdata) {
        // Skip for logged in users with edit capability
        if (current_user_can('edit_posts')) {
            return $commentdata;
        }

        // Build data array for analysis
        $data = [
            'comment_author' => $commentdata['comment_author'] ?? '',
            'comment_author_email' => $commentdata['comment_author_email'] ?? '',
            'comment_author_url' => $commentdata['comment_author_url'] ?? '',
            'comment_content' => $commentdata['comment_content'] ?? '',
        ];

        $result = $this->analyze_submission($data, 'wp-comment');

        if ($result['is_spam']) {
            wp_die(
                esc_html($this->get_translated_spam_message()),
                esc_html__('Comment Blocked', 'honeypot-guard-silent-anti-spam'),
                ['response' => 403, 'back_link' => true]
            );
        }

        return $commentdata;
    }

    /**
     * bbPress Forum spam check
     */
    public function check_bbpress_spam($content) {
        // Skip for logged in users with moderate capability
        if (current_user_can('moderate')) {
            return $content;
        }

        // Build data array including POST data
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- bbPress handles nonce verification
        $data = [
            'content' => $content,
            'title' => isset($_POST['bbp_topic_title']) ? sanitize_text_field(wp_unslash($_POST['bbp_topic_title'])) : '',
            'author' => isset($_POST['bbp_anonymous_name']) ? sanitize_text_field(wp_unslash($_POST['bbp_anonymous_name'])) : '',
            'email' => isset($_POST['bbp_anonymous_email']) ? sanitize_email(wp_unslash($_POST['bbp_anonymous_email'])) : '',
        ];
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        $result = $this->analyze_submission($data, 'bbpress');

        if ($result['is_spam']) {
            wp_die(
                esc_html($this->get_translated_spam_message()),
                esc_html__('Submission Blocked', 'honeypot-guard-silent-anti-spam'),
                ['response' => 403, 'back_link' => true]
            );
        }

        return $content;
    }

    /**
     * =========================================================================
     * REGISTRATION FORM PROTECTION
     * =========================================================================
     */

    /**
     * Add honeypot fields to registration forms
     * Works with WordPress, WooCommerce, and Shopify-style forms
     */
    public function add_registration_honeypot_fields() {
        if (empty($this->options['protect_registration_forms'])) {
            return;
        }

        $field_name = $this->get_honeypot_field_name();
        $nonce = wp_create_nonce('honeypot_guard_registration');
        $timestamp = time();

        // Hidden honeypot field (bots will fill this)
        echo '<div style="position:absolute;left:-9999px;top:-9999px;visibility:hidden;height:0;width:0;overflow:hidden;opacity:0;pointer-events:none;" aria-hidden="true">';
        echo '<label for="' . esc_attr($field_name) . '">' . esc_html__('Leave empty', 'honeypot-guard-silent-anti-spam') . '</label>';
        echo '<input type="text" name="' . esc_attr($field_name) . '" id="' . esc_attr($field_name) . '" value="" tabindex="-1" autocomplete="new-password" aria-hidden="true" />';
        echo '</div>';

        // Timestamp field
        echo '<input type="hidden" name="_oh_reg_timestamp" value="' . esc_attr($timestamp) . '" />';
        // Nonce field
        echo '<input type="hidden" name="_oh_reg_nonce" value="' . esc_attr($nonce) . '" />';
    }

    /**
     * Check WordPress registration for spam
     *
     * @param WP_Error $errors Registration errors
     * @param string $sanitized_user_login User login
     * @param string $user_email User email
     * @return WP_Error
     */
    public function check_registration_spam($errors, $sanitized_user_login, $user_email) {
        if (empty($this->options['protect_registration_forms'])) {
            return $errors;
        }

        // Build data array for analysis
        $data = [
            'user_login' => $sanitized_user_login,
            'user_email' => $user_email,
        ];

        // Add honeypot field data
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- WordPress handles nonce in registration
        $honeypot_field = $this->get_honeypot_field_name();
        if (isset($_POST[$honeypot_field])) {
            $data[$honeypot_field] = sanitize_text_field(wp_unslash($_POST[$honeypot_field]));
        }

        // Add timestamp
        if (isset($_POST['_oh_reg_timestamp'])) {
            $data['_oh_timestamp'] = sanitize_text_field(wp_unslash($_POST['_oh_reg_timestamp']));
        }
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        $result = $this->analyze_submission($data, 'wp-registration');

        if ($result['is_spam']) {
            $errors->add('spam_error', $this->get_translated_spam_message());
        } else {
            // Fire registration conversion events on successful check
            $this->fire_registration_conversion_events('wp-registration', $sanitized_user_login, $user_email);
        }

        return $errors;
    }

    /**
     * Check WooCommerce registration for spam
     *
     * @param WP_Error $errors Registration errors
     * @param string $username Username
     * @param string $email Email
     * @return WP_Error
     */
    public function check_woocommerce_registration_spam($errors, $username, $email) {
        if (empty($this->options['protect_registration_forms'])) {
            return $errors;
        }

        // Build data array for analysis
        $data = [
            'user_login' => $username,
            'user_email' => $email,
        ];

        // Add honeypot field data
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- WooCommerce handles nonce in registration
        $honeypot_field = $this->get_honeypot_field_name();
        if (isset($_POST[$honeypot_field])) {
            $data[$honeypot_field] = sanitize_text_field(wp_unslash($_POST[$honeypot_field]));
        }

        // Add timestamp
        if (isset($_POST['_oh_reg_timestamp'])) {
            $data['_oh_timestamp'] = sanitize_text_field(wp_unslash($_POST['_oh_reg_timestamp']));
        }

        // Add any additional WooCommerce fields
        if (isset($_POST['billing_first_name'])) {
            $data['first_name'] = sanitize_text_field(wp_unslash($_POST['billing_first_name']));
        }
        if (isset($_POST['billing_last_name'])) {
            $data['last_name'] = sanitize_text_field(wp_unslash($_POST['billing_last_name']));
        }
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        $result = $this->analyze_submission($data, 'woocommerce-registration');

        if ($result['is_spam']) {
            $errors->add('spam_error', $this->get_translated_spam_message());
        } else {
            // Fire registration conversion events on successful check
            $this->fire_registration_conversion_events('woocommerce-registration', $username, $email);
        }

        return $errors;
    }

    /**
     * Fire conversion events for successful registration
     * Only fires when protect_registration_forms is enabled
     *
     * @param string $source Registration source (wp-registration, woocommerce-registration)
     * @param string $username Username
     * @param string $email Email
     */
    private function fire_registration_conversion_events($source, $username, $email) {
        if (empty($this->options['protect_registration_forms'])) {
            return;
        }

        // Store data for frontend script to fire events
        // This will be picked up by the honeypot script in footer
        if (!isset($GLOBALS['honeypot_guard_registration_success'])) {
            $GLOBALS['honeypot_guard_registration_success'] = [];
        }
        $GLOBALS['honeypot_guard_registration_success'][] = [
            'source' => $source,
            'username' => $username,
            'email_domain' => substr(strrchr($email, "@"), 1),
        ];
    }

    /**
     * Callback for WordPress user_register action
     * Logs successful registration and prepares conversion event data
     *
     * @param int $user_id User ID
     * @param array $userdata User data (WP 5.8+)
     */
    public function on_user_registered($user_id, $userdata = []) {
        if (empty($this->options['protect_registration_forms'])) {
            return;
        }

        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }

        // Log the successful registration
        $this->log_submission([
            'user_login' => $user->user_login,
            'user_email' => $user->user_email,
        ], 'wp-registration', false, 0, []);

        // Set a transient to fire conversion events on next page load
        set_transient('honeypot_guard_registration_success_' . $user_id, [
            'source' => 'wp-registration',
            'user_id' => $user_id,
            'username' => $user->user_login,
        ], 60);
    }

    /**
     * Callback for WooCommerce woocommerce_created_customer action
     * Logs successful customer creation and prepares conversion event data
     *
     * @param int $customer_id Customer/User ID
     * @param array $new_customer_data Customer data
     * @param bool $password_generated Whether password was generated
     */
    public function on_woocommerce_customer_created($customer_id, $new_customer_data, $password_generated) {
        if (empty($this->options['protect_registration_forms'])) {
            return;
        }

        // Log the successful registration
        $this->log_submission([
            'user_login' => $new_customer_data['user_login'] ?? '',
            'user_email' => $new_customer_data['user_email'] ?? '',
        ], 'woocommerce-registration', false, 0, []);

        // Set a transient to fire conversion events on next page load
        set_transient('honeypot_guard_registration_success_' . $customer_id, [
            'source' => 'woocommerce-registration',
            'user_id' => $customer_id,
            'username' => $new_customer_data['user_login'] ?? '',
        ], 60);
    }

    /**
     * Output registration conversion events script
     * Fires FB Pixel and GTM events after successful registration
     */
    public function enqueue_registration_conversion_script() {
        if (empty($this->options['protect_registration_forms'])) {
            return;
        }

        // Check if current user just registered
        if (!is_user_logged_in()) {
            return;
        }

        $user_id = get_current_user_id();
        $transient_key = 'honeypot_guard_registration_success_' . $user_id;
        $registration_data = get_transient($transient_key);

        if (!$registration_data) {
            return;
        }

        // Delete the transient so we don't fire events again
        delete_transient($transient_key);

        $source = $registration_data['source'] ?? 'registration';

        // Register a dummy handle for the inline script
        wp_register_script('honeypot-guard-reg-conversion', false, [], HONEYPOT_GUARD_VERSION, true);
        wp_enqueue_script('honeypot-guard-reg-conversion');
        wp_add_inline_script('honeypot-guard-reg-conversion', '(function(){
            "use strict";
            if(window._ohFbq){window._ohFbq.allow();}
            if(window.fbq){
                try{
                    window.fbq("track","CompleteRegistration",{content_name:"' . esc_js($source) . '",status:"success"});
                    window.fbq("track","Lead");
                }catch(e){}
            }
            if(window.dataLayer){
                try{
                    window.dataLayer.push({"event":"registration","registration_source":"' . esc_js($source) . '","registration_status":"success"});
                    window.dataLayer.push({"event":"sign_up","method":"' . esc_js($source) . '"});
                    window.dataLayer.push({"event":"generate_lead","lead_source":"' . esc_js($source) . '"});
                }catch(e){}
            }
        })();');
    }

    /**
     * Generic form check
     */
    public function check_generic_form_submission() {
        if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'POST' || is_admin()) {
            return;
        }

        // Skip if this is a known form plugin submission - let their handlers deal with it
        if (isset($_POST['_wpcf7']) || isset($_POST['_wpcf7_unit_tag'])) {
            return; // Contact Form 7
        }
        if (isset($_POST['wpforms'])) {
            return; // WPForms
        }
        if (isset($_POST['gform_submit'])) {
            return; // Gravity Forms
        }
        if (isset($_POST['_ninja_forms_display_submit'])) {
            return; // Ninja Forms
        }

        // Only check if honeypot field is present
        $honeypot_field = $this->get_honeypot_field_name();
        if (!isset($_POST[$honeypot_field]) && !isset($_POST['_oh_timestamp'])) {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Generic form handler, nonce verified by form plugin
        $result = $this->analyze_submission($this->sanitize_submission_data($this->get_relevant_post_fields()), 'generic-form');

        if ($result['is_spam']) {
            status_header(403);
            wp_die(
                esc_html($this->get_translated_spam_message()),
                esc_html__('Submission Blocked', 'honeypot-guard-silent-anti-spam'),
                ['response' => 403, 'back_link' => true]
            );
        }
    }

    /**
     * Get spam rejection message (simple version - delegates to translated)
     */
    private function get_spam_message() {
        return $this->get_translated_spam_message();
    }

    /**
     * Enqueue frontend scripts (fbq interceptor + honeypot protection)
     */
    public function enqueue_frontend_scripts() {
        // Fbq interceptor - enqueue in head with highest priority
        wp_register_script('honeypot-guard-fbq-block', false, [], HONEYPOT_GUARD_VERSION, false);
        wp_enqueue_script('honeypot-guard-fbq-block');
        wp_add_inline_script('honeypot-guard-fbq-block', $this->get_fbq_interceptor_inline_js());

        // Honeypot protection - enqueue in footer
        wp_register_script('honeypot-guard-protection', false, [], HONEYPOT_GUARD_VERSION, true);
        wp_enqueue_script('honeypot-guard-protection');
        wp_add_inline_script('honeypot-guard-protection', $this->get_honeypot_protection_inline_js());
    }

    /**
     * Get the fbq + dataLayer interceptor inline JS (no script tags)
     */
    private function get_fbq_interceptor_inline_js() {
        $protect_registration = !empty($this->options['protect_registration_forms']) ? 'true' : 'false';
        return '(function(w,d){
if(w._ohFbq)return;
var protectReg=' . $protect_registration . ';
var F=["form_submit","form_start","submitapplication"];
var R=["completeregistration","sign_up","signup","create_account","register"];
var E=["form_submit","form_start","contact_form","form_submission","wpcf7","cf7","comment","submitapplication"];
var RE=["registration","sign_up","signup","create_account","register","new_user","customer_created"];
if(protectReg){F=F.concat(R);E=E.concat(RE);}
var ok=false,real=null,q=[];
function hasFB(s){return s&&(s.indexOf("facebook.com/tr")!==-1||s.indexOf("facebook.net/tr")!==-1);}
function hasFormEvt(s){
if(!s)return false;
try{s=decodeURIComponent(s);}catch(e){}
s=s.toLowerCase();
for(var i=0;i<F.length;i++)if(s.indexOf("ev="+F[i])!==-1||s.indexOf("&e="+F[i])!==-1||s.indexOf("event="+F[i])!==-1)return true;
return false;
}
function shouldBlock(url){return!ok&&hasFB(url)&&hasFormEvt(url);}
var imgProto=HTMLImageElement.prototype;
var imgSrcDesc=Object.getOwnPropertyDescriptor(imgProto,"src");
if(imgSrcDesc&&imgSrcDesc.set){
Object.defineProperty(imgProto,"src",{
configurable:true,enumerable:true,
get:imgSrcDesc.get,
set:function(v){if(shouldBlock(v))return;imgSrcDesc.set.call(this,v);}
});
}
var oXHR=w.XMLHttpRequest.prototype.open;
w.XMLHttpRequest.prototype.open=function(){
var url=arguments[1]||"";
if(shouldBlock(url)){this._ohBlocked=true;return;}
this._ohBlocked=false;
return oXHR.apply(this,arguments);
};
var oSend=w.XMLHttpRequest.prototype.send;
w.XMLHttpRequest.prototype.send=function(b){
if(this._ohBlocked)return;
return oSend.apply(this,arguments);
};
if(navigator.sendBeacon){
var oBeacon=navigator.sendBeacon.bind(navigator);
navigator.sendBeacon=function(u,d){if(shouldBlock(u))return true;return oBeacon(u,d);};
}
if(w.fetch){
var oFetch=w.fetch.bind(w);
w.fetch=function(u,o){
var url=typeof u==="string"?u:(u&&u.url?u.url:"");
if(shouldBlock(url))return Promise.resolve(new Response("",{status:200}));
return oFetch(u,o);
};
}
function isBlocked(a){
var cmd=a[0],evt=(a[1]||"").toLowerCase();
return(cmd==="track"||cmd==="trackCustom")&&F.indexOf(evt)!==-1&&!ok;
}
function P(){
var a=[].slice.call(arguments);
if(isBlocked(a))return;
if(real)return real.apply(w,a);
q.push(a);
}
P._ohP=1;P.queue=[];P.loaded=1;P.version="2.0";
P.push=P.callMethod=function(){P.apply(w,arguments);};
P.getState=function(){return P;};
function setReal(f){if(!f||f._ohP||f===P)return;real=f;for(var k in f)if(f.hasOwnProperty(k)&&!P[k])try{P[k]=f[k];}catch(e){}for(var i=0;i<q.length;i++){var c=q[i];if(!isBlocked(c))try{f.apply(w,c);}catch(e){}}q=[];}
w.fbq=w._fbq=P;
try{
Object.defineProperty(w,"fbq",{configurable:1,enumerable:1,get:function(){return P;},set:setReal});
Object.defineProperty(w,"_fbq",{configurable:1,enumerable:1,get:function(){return P;},set:setReal});
}catch(e){}
function isFormEvt(o){
if(!o||typeof o!=="object")return false;
var ev=(o.event||o.eventName||"").toLowerCase();
for(var i=0;i<E.length;i++)if(ev.indexOf(E[i])!==-1)return true;
var cat=(o.event_category||o.eventCategory||"").toLowerCase();
return cat&&(cat.indexOf("form")!==-1||cat.indexOf("contact")!==-1||cat.indexOf("comment")!==-1);
}
var dL=w.dataLayer=w.dataLayer||[];
if(!dL._ohWrap){
dL._ohWrap=1;
var oP=[].push;
dL.push=function(){
var r=[];for(var i=0;i<arguments.length;i++){var o=arguments[i];if(!isFormEvt(o)||ok)r.push(o);}
return r.length?oP.apply(dL,r):dL.length;
};
}
w._ohFbq={allow:function(){ok=1;setTimeout(function(){ok=0;},3000);},reset:function(){ok=0;}};
})(window,document);';
    }

    /**
     * Get honeypot protection inline JS (no script tags)
     */
    private function get_honeypot_protection_inline_js() {
        if (!$this->options['honeypot_enabled']) {
            return '';
        }

        // Skip if JS injection is disabled
        if (empty($this->options['honeypot_inject_js'])) {
            return '';
        }

        $field_name = $this->get_honeypot_field_name();
        $nonce = wp_create_nonce('honeypot_guard_form');
        $use_multiple = !empty($this->options['honeypot_multiple']);
        $additional_fields = '';
        $cookie_check = !empty($this->options['cookie_check_enabled']) ? '1' : '0';

        if ($use_multiple) {
            $extra_names = $this->get_honeypot_field_names(2);
            $additional_fields = implode(',', array_slice($extra_names, 0, 2));
        }

        $js = '(function(){'
            . '"use strict";'
            . 'var oh={'
            . 'fn:' . wp_json_encode($field_name) . ','
            . 'n:' . wp_json_encode($nonce) . ','
            . 'mf:' . wp_json_encode($additional_fields) . ','
            . 'cc:' . wp_json_encode($cookie_check) . ','
            . 'q:[],'
            . 'r:false,'
            . 'init:function(){'
            . 'if(document.readyState==="loading"){'
            . 'document.addEventListener("DOMContentLoaded",function(){'
            . 'requestAnimationFrame(function(){oh.setup();});'
            . '});'
            . '}else{'
            . 'requestAnimationFrame(function(){oh.setup();});'
            . '}'
            . '},'
            . 'setup:function(){'
            . 'if(oh.cc==="1"){oh.setCookie();}'
            . 'var forms=document.querySelectorAll("form:not([data-oh])");'
            . 'for(var i=0;i<forms.length;i++){oh.protect(forms[i]);}'
            . 'oh.observe();'
            . 'oh.listenCF7();'
            . 'oh.listenPopups();'
            . 'oh.r=true;'
            . '},'
            . 'setCookie:function(){'
            . 'var d=new Date();'
            . 'd.setTime(d.getTime()+(24*60*60*1000));'
            . 'var v=oh.n.substring(0,8)+"-"+Math.random().toString(36).substring(2,10);'
            . 'document.cookie="_oh_verify="+v+";expires="+d.toUTCString()+";path=/;SameSite=Lax";'
            . '},'
            . 'protect:function(f){'
            . 'if(!f||f.hasAttribute("data-oh"))return;'
            . 'if(f.classList.contains("oh-exclude")||f.classList.contains("gm-exclude")||f.hasAttribute("data-oh-exclude"))return;'
            . 'f.setAttribute("data-oh","1");'
            . 'oh.addHoneypot(f,oh.fn);'
            . 'if(oh.mf){'
            . 'var extra=oh.mf.split(",");'
            . 'for(var i=0;i<extra.length;i++){if(extra[i])oh.addHoneypot(f,extra[i]);}'
            . '}'
            . 'if(!f.querySelector("[name=\\"_oh_timestamp\\"]")){'
            . 'var ts=document.createElement("input");'
            . 'ts.type="hidden";ts.name="_oh_timestamp";ts.value=Math.floor(Date.now()/1000);'
            . 'f.appendChild(ts);'
            . '}'
            . 'if(!f.querySelector("[name=\\"_oh_nonce\\"]")){'
            . 'var nc=document.createElement("input");'
            . 'nc.type="hidden";nc.name="_oh_nonce";nc.value=oh.n;'
            . 'f.appendChild(nc);'
            . '}'
            . '},'
            . 'addHoneypot:function(f,name){'
            . 'if(f.querySelector("[name=\\""+name+"\\"]"))return;'
            . 'var w=document.createElement("div");'
            . 'w.style.cssText="position:absolute!important;left:-9999px!important;top:-9999px!important;visibility:hidden!important;height:0!important;width:0!important;overflow:hidden!important;opacity:0!important;pointer-events:none!important;";'
            . 'w.setAttribute("aria-hidden","true");'
            . 'var inp=document.createElement("input");'
            . 'inp.type="text";inp.name=name;inp.tabIndex=-1;inp.autocomplete="new-password";'
            . 'inp.setAttribute("aria-hidden","true");'
            . 'w.appendChild(inp);'
            . 'f.appendChild(w);'
            . '},'
            . 'observe:function(){'
            . 'if(!window.MutationObserver)return;'
            . 'var timer=null;'
            . 'var obs=new MutationObserver(function(m){'
            . 'if(timer)clearTimeout(timer);'
            . 'timer=setTimeout(function(){'
            . 'm.forEach(function(mut){'
            . 'mut.addedNodes.forEach(function(n){'
            . 'if(n.nodeType===1){'
            . 'if(n.tagName==="FORM")oh.protect(n);'
            . 'else if(n.querySelectorAll){'
            . 'var fs=n.querySelectorAll("form:not([data-oh])");'
            . 'for(var i=0;i<fs.length;i++)oh.protect(fs[i]);'
            . '}'
            . '}'
            . '});'
            . '});'
            . '},50);'
            . '});'
            . 'obs.observe(document.body,{childList:true,subtree:true});'
            . '},'
            . 'listenCF7:function(){'
            . 'var cf7Fired={};'
            . 'function fireConversions(formId){'
            . 'var key=formId+"-"+Date.now();'
            . 'if(cf7Fired[formId]&&(Date.now()-cf7Fired[formId])<5000)return;'
            . 'cf7Fired[formId]=Date.now();'
            . 'if(window._ohFbq){window._ohFbq.allow();}'
            . 'setTimeout(function(){'
            . 'if(window.fbq){'
            . 'try{window.fbq("track","Lead");}catch(ex){}'
            . 'try{window.fbq("track","Contact");}catch(ex){}'
            . 'try{window.fbq("track","SubmitApplication");}catch(ex){}'
            . 'try{window.fbq("trackCustom","form_submit",{form_id:formId||""});}catch(ex){}'
            . '}'
            . 'if(window.dataLayer){'
            . 'try{'
            . 'window.dataLayer.push({"event":"form_submit","form_id":formId||"","form_status":"success"});'
            . 'window.dataLayer.push({"event":"generate_lead","form_id":formId||""});'
            . '}catch(ex){}'
            . '}'
            . '},50);'
            . '}'
            . 'function handleCF7MailSent(e){'
            . 'var detail=e.detail||{};'
            . 'var formId=detail.contactFormId||detail.id||"";'
            . 'fireConversions(formId);'
            . '}'
            . 'function handleCF7Submit(e){'
            . 'var detail=e.detail||{};'
            . 'var status=detail.status||"";'
            . 'if(status==="mail_sent"){'
            . 'var formId=detail.contactFormId||detail.id||"";'
            . 'fireConversions(formId);'
            . '}else if(window._ohFbq){'
            . 'window._ohFbq.reset();'
            . '}'
            . '}'
            . 'document.addEventListener("wpcf7mailsent",handleCF7MailSent,false);'
            . 'document.addEventListener("wpcf7:mailsent",handleCF7MailSent,false);'
            . 'document.addEventListener("wpcf7submit",handleCF7Submit,false);'
            . 'document.addEventListener("wpcf7:submit",handleCF7Submit,false);'
            . 'document.addEventListener("wpcf7:init",function(e){'
            . 'if(e.detail&&e.detail.container){var f=e.detail.container.querySelector("form");if(f)oh.protect(f);}'
            . '});'
            . 'document.addEventListener("wpcf7:mailsent",function(){setTimeout(function(){oh.scanAll();},100);});'
            . 'document.addEventListener("wpcf7:invalid",function(){setTimeout(function(){oh.scanAll();},100);});'
            . '},'
            . 'listenPopups:function(){'
            . 'if(typeof elementorFrontend!=="undefined"){'
            . 'jQuery(document).on("elementor/popup/show",function(){setTimeout(function(){oh.scanAll();},100);});'
            . '}'
            . 'document.addEventListener("popup:opened",function(){oh.scanAll();});'
            . 'document.addEventListener("modal:opened",function(){oh.scanAll();});'
            . 'if(typeof PUM!=="undefined"){jQuery(document).on("pumAfterOpen",function(){setTimeout(function(){oh.scanAll();},100);});}'
            . 'jQuery(document).on("dialogopen",function(){setTimeout(function(){oh.scanAll();},100);});'
            . 'jQuery(document).on("shown.bs.modal",function(){setTimeout(function(){oh.scanAll();},100);});'
            . 'jQuery(document).on("afterShow.fb",function(){setTimeout(function(){oh.scanAll();},100);});'
            . 'document.addEventListener("transitionend",function(e){'
            . 'if(e.target&&e.target.querySelectorAll){'
            . 'var fs=e.target.querySelectorAll("form:not([data-oh])");'
            . 'for(var i=0;i<fs.length;i++)oh.protect(fs[i]);'
            . '}'
            . '});'
            . '},'
            . 'scanAll:function(){'
            . 'var forms=document.querySelectorAll("form:not([data-oh])");'
            . 'for(var i=0;i<forms.length;i++){oh.protect(forms[i]);}'
            . '}'
            . '};'
            . 'oh.init();'
            . 'window.HoneypotGuard=oh;'
            . '})();';

        return $js;
    }

    /**
     * =========================================================================
     * REST API
     * =========================================================================
     */

    /**
     * Register REST routes
     */
    public function register_rest_routes() {
        register_rest_route('honeypot-guard/v1', '/validate', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_validate'],
            'permission_callback' => function($request) {
                // Verify nonce to ensure request originates from a legitimate page load
                $nonce = $request->get_header('X-WP-Nonce');
                if ($nonce && wp_verify_nonce($nonce, 'wp_rest')) {
                    return true;
                }
                // Also accept if a valid honeypot nonce is present in the data
                $data = $request->get_param('data');
                if (is_array($data) && !empty($data['_oh_nonce']) && wp_verify_nonce($data['_oh_nonce'], 'honeypot_guard_form')) {
                    return true;
                }
                return new WP_Error('rest_forbidden', __('Nonce verification failed.', 'honeypot-guard-silent-anti-spam'), ['status' => 403]);
            },
            'args' => [
                'data' => ['required' => true, 'type' => 'object'],
            ]
        ]);

        register_rest_route('honeypot-guard/v1', '/config', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_config'],
            'permission_callback' => function($request) {
                // Verify nonce to ensure request originates from a legitimate page load
                // This protects honeypot config from being scraped by attackers
                $nonce = $request->get_header('X-WP-Nonce');
                if ($nonce && wp_verify_nonce($nonce, 'wp_rest')) {
                    return true;
                }
                // Also accept honeypot nonce from query param for frontend script requests
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce is being verified here
                $param_nonce = isset($_GET['_nonce']) ? sanitize_text_field(wp_unslash($_GET['_nonce'])) : '';
                if ($param_nonce && wp_verify_nonce($param_nonce, 'honeypot_guard_form')) {
                    return true;
                }
                return new WP_Error('rest_forbidden', __('Nonce verification failed.', 'honeypot-guard-silent-anti-spam'), ['status' => 403]);
            },
        ]);

        register_rest_route('honeypot-guard/v1', '/stats', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_stats'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            },
        ]);
    }

    /**
     * REST: Validate submission
     */
    public function rest_validate($request) {
        $data = $request->get_param('data');

        if (empty($data) || !is_array($data)) {
            return new WP_REST_Response(['error' => 'Invalid data'], 400);
        }

        $result = $this->analyze_submission($data, 'rest-api');

        return new WP_REST_Response([
            'is_spam' => $result['is_spam'],
            'score' => $result['score'],
            'reasons' => $result['reasons']
        ], 200);
    }

    /**
     * REST: Get config for JS
     */
    public function rest_config($request) {
        return new WP_REST_Response([
            'honeypot_field' => $this->get_honeypot_field_name(),
            'min_time' => $this->options['min_submission_time'],
            'max_time' => $this->options['max_submission_time'],
            'version' => HONEYPOT_GUARD_VERSION
        ], 200);
    }

    /**
     * REST: Get stats (admin only)
     */
    public function rest_stats($request) {
        global $wpdb;

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table stats
        return new WP_REST_Response([
            'total_spam' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam'"),
            'total_clean' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status='clean'"),
            'today_spam' => (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam' AND created_at >= %s",
                gmdate('Y-m-d 00:00:00')
            )),
            'blacklist_count' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->blacklist_table}")
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        ], 200);
    }

    /**
     * =========================================================================
     * ADMIN INTERFACE
     * =========================================================================
     */

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            honeypot_guard_t('plugin_name'),
            honeypot_guard_t('plugin_name'),
            'manage_options',
            'honeypot-guard',
            [$this, 'render_dashboard'],
            'dashicons-shield-alt',
            80
        );

        add_submenu_page(
            'honeypot-guard',
            honeypot_guard_t('dashboard'),
            honeypot_guard_t('dashboard'),
            'manage_options',
            'honeypot-guard',
            [$this, 'render_dashboard']
        );

        add_submenu_page(
            'honeypot-guard',
            honeypot_guard_t('settings'),
            honeypot_guard_t('settings'),
            'manage_options',
            'honeypot-guard-settings',
            [$this, 'render_settings']
        );

        add_submenu_page(
            'honeypot-guard',
            honeypot_guard_t('blacklists'),
            honeypot_guard_t('blacklists'),
            'manage_options',
            'honeypot-guard-blacklist',
            [$this, 'render_blacklist']
        );

        add_submenu_page(
            'honeypot-guard',
            honeypot_guard_t('spam_logs'),
            honeypot_guard_t('spam_logs'),
            'manage_options',
            'honeypot-guard-logs',
            [$this, 'render_logs']
        );
    }

    /**
     * Add plugin action links
     */
    public function add_plugin_links($links) {
        $settings_link = '<a href="' . esc_url(admin_url('admin.php?page=honeypot-guard-settings')) . '">' . esc_html(honeypot_guard_t('settings')) . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('honeypot_guard_options', 'honeypot_guard_options', [
            'sanitize_callback' => [$this, 'sanitize_options']
        ]);
    }

    /**
     * Sanitize options
     */
    public function sanitize_options($input) {
        $sanitized = [];

        // Language
        $valid_languages = array_keys(HoneypotGuard_Translations::$languages);
        $sanitized['plugin_language'] = in_array($input['plugin_language'] ?? 'en', $valid_languages)
            ? $input['plugin_language'] : 'en';

        // Honeypot
        $sanitized['honeypot_enabled'] = !empty($input['honeypot_enabled']);
        $sanitized['honeypot_field_name'] = sanitize_key($input['honeypot_field_name'] ?? 'middle_name');
        $sanitized['honeypot_mode'] = in_array($input['honeypot_mode'] ?? 'auto', ['auto', 'dynamic', 'custom'])
            ? $input['honeypot_mode'] : 'auto';
        $sanitized['honeypot_rotation'] = in_array($input['honeypot_rotation'] ?? 'session', ['session', 'hourly', 'daily'])
            ? $input['honeypot_rotation'] : 'session';
        $sanitized['honeypot_inject_js'] = !empty($input['honeypot_inject_js']);
        $sanitized['honeypot_multiple'] = !empty($input['honeypot_multiple']);
        $sanitized['honeypot_custom_names'] = sanitize_textarea_field($input['honeypot_custom_names'] ?? '');
        $sanitized['auto_generate_name'] = !empty($input['auto_generate_name']);

        // Timestamp
        $sanitized['timestamp_enabled'] = !empty($input['timestamp_enabled']);
        $sanitized['min_submission_time'] = absint($input['min_submission_time'] ?? 2);
        $sanitized['max_submission_time'] = absint($input['max_submission_time'] ?? 3600);

        // Character filters
        $sanitized['cyrillic_filter'] = !empty($input['cyrillic_filter']);
        $sanitized['hanzi_filter'] = !empty($input['hanzi_filter']);
        $sanitized['bengali_filter'] = !empty($input['bengali_filter']);
        $sanitized['latin_filter'] = !empty($input['latin_filter']);

        // Blacklists
        $sanitized['blacklist_emails'] = sanitize_textarea_field($input['blacklist_emails'] ?? '');
        $sanitized['blacklist_domains'] = sanitize_textarea_field($input['blacklist_domains'] ?? '');
        $sanitized['blacklist_keywords_subject'] = sanitize_textarea_field($input['blacklist_keywords_subject'] ?? '');
        $sanitized['blacklist_keywords_body'] = sanitize_textarea_field($input['blacklist_keywords_body'] ?? '');
        $sanitized['blacklist_keywords_all'] = sanitize_textarea_field($input['blacklist_keywords_all'] ?? '');
        $sanitized['blacklist_ips'] = sanitize_textarea_field($input['blacklist_ips'] ?? '');

        // Heuristics
        $sanitized['heuristic_enabled'] = !empty($input['heuristic_enabled']);
        $sanitized['max_links'] = absint($input['max_links'] ?? 3);
        $sanitized['spam_threshold'] = min(200, max(1, absint($input['spam_threshold'] ?? 70)));

        // Rate limiting
        $sanitized['rate_limit_enabled'] = !empty($input['rate_limit_enabled']);
        $sanitized['rate_limit_submissions'] = absint($input['rate_limit_submissions'] ?? 5);
        $sanitized['rate_limit_window'] = absint($input['rate_limit_window'] ?? 60);

        // Security
        $sanitized['require_referer'] = !empty($input['require_referer']);
        $sanitized['cookie_check_enabled'] = !empty($input['cookie_check_enabled']);

        // Logging
        $sanitized['log_retention_days'] = absint($input['log_retention_days'] ?? 30);
        $sanitized['log_clean_submissions'] = !empty($input['log_clean_submissions']);

        // Escalation
        $sanitized['escalation_threshold'] = absint($input['escalation_threshold'] ?? 3);
        $sanitized['escalation_window'] = absint($input['escalation_window'] ?? 60);

        // AI Gibberish Detection
        $sanitized['gibberish_detection_enabled'] = !empty($input['gibberish_detection_enabled']);
        $sanitized['gibberish_ai_provider'] = in_array($input['gibberish_ai_provider'] ?? 'openai', ['openai', 'anthropic'])
            ? $input['gibberish_ai_provider'] : 'openai';
        $sanitized['gibberish_api_key'] = sanitize_text_field($input['gibberish_api_key'] ?? '');
        $sanitized['gibberish_language'] = sanitize_key($input['gibberish_language'] ?? 'auto');
        $sanitized['gibberish_domain_languages'] = sanitize_textarea_field($input['gibberish_domain_languages'] ?? '');
        $sanitized['gibberish_score'] = min(100, max(1, absint($input['gibberish_score'] ?? 85)));
        $sanitized['gibberish_min_length'] = min(50, max(3, absint($input['gibberish_min_length'] ?? 5)));

        // Spam Messages
        $sanitized['spam_message_mode'] = in_array($input['spam_message_mode'] ?? 'auto', ['auto', 'custom', 'per_domain'])
            ? $input['spam_message_mode'] : 'auto';
        $sanitized['custom_spam_message'] = sanitize_text_field($input['custom_spam_message'] ?? '');
        $sanitized['spam_messages_per_domain'] = sanitize_textarea_field($input['spam_messages_per_domain'] ?? '');
        $sanitized['spam_message_default'] = sanitize_text_field($input['spam_message_default'] ?? '');

        // Advanced
        $sanitized['whitelist_logged_in'] = !empty($input['whitelist_logged_in']);
        $sanitized['whitelist_roles'] = isset($input['whitelist_roles']) && is_array($input['whitelist_roles'])
            ? array_map('sanitize_key', $input['whitelist_roles'])
            : ['administrator', 'editor'];
        $sanitized['delete_data_on_uninstall'] = !empty($input['delete_data_on_uninstall']);

        // Registration Form Protection
        $sanitized['protect_registration_forms'] = !empty($input['protect_registration_forms']);

        // Clear caches when options are saved
        $this->clear_options_cache();

        return $sanitized;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'honeypot-guard') === false) {
            return;
        }

        // Initialize translator with saved language
        $this->init_translator();

        wp_enqueue_style(
            'honeypot-guard-admin',
            HONEYPOT_GUARD_PLUGIN_URL . 'assets/css/admin.css',
            [],
            HONEYPOT_GUARD_VERSION
        );

        wp_enqueue_script(
            'honeypot-guard-admin',
            HONEYPOT_GUARD_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery'],
            HONEYPOT_GUARD_VERSION,
            true
        );

        wp_localize_script('honeypot-guard-admin', 'honeypotGuardAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('honeypot_guard_admin'),
            'landingUrl' => self::LANDING_PAGE_URL,
            'supportUrl' => self::SUPPORT_URL,
            'strings' => [
                'confirmDelete' => honeypot_guard_t('confirm_delete'),
                'confirmClear' => honeypot_guard_t('confirm_clear_logs'),
                'deleted' => honeypot_guard_t('entry_deleted'),
                'cleared' => honeypot_guard_t('logs_cleared'),
                'added' => honeypot_guard_t('added_to_blacklist'),
                'error' => honeypot_guard_t('error_occurred'),
                // Additional strings for inline scripts
                'confirmBlacklistIp' => honeypot_guard_t('confirm_blacklist_ip'),
                'confirmUnblacklistIp' => honeypot_guard_t('confirm_unblacklist_ip'),
                'addedFromSpamLogs' => honeypot_guard_t('added_from_spam_logs'),
                'removeIpFromBlacklist' => honeypot_guard_t('remove_ip_from_blacklist'),
                'addIpToBlacklist' => honeypot_guard_t('add_ip_to_blacklist'),
                'enterValueToBlock' => honeypot_guard_t('enter_value_to_block'),
                'hideFilters' => honeypot_guard_t('hide_filters') ?: 'Hide Filters',
                'showFilters' => honeypot_guard_t('show_filters') ?: 'Show Filters'
            ]
        ]);
    }

    /**
     * Initialize translator with saved language
     */
    public function init_translator() {
        $translator = HoneypotGuard_Translations::get_instance();
        $language = $this->options['plugin_language'] ?? 'en';
        $translator->set_language($language);
    }

    /**
     * Render dashboard
     */
    public function render_dashboard() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', 'honeypot-guard-silent-anti-spam'));
        }

        $this->init_translator();
        global $wpdb;

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table stats
        $total_spam = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam'");
        $total_clean = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status='clean'");
        $spam_today = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam' AND created_at >= %s",
            gmdate('Y-m-d 00:00:00')
        ));
        $spam_last_hour = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam' AND created_at >= %s",
            gmdate('Y-m-d H:i:s', strtotime('-1 hour'))
        ));
        $blacklist_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->blacklist_table}");
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        $show_escalation = $spam_last_hour >= $this->options['escalation_threshold'];

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query
        $recent_logs = $wpdb->get_results(
            "SELECT * FROM {$this->table_name} ORDER BY created_at DESC LIMIT 10"
        );

        include HONEYPOT_GUARD_PLUGIN_DIR . 'views/dashboard.php';
    }

    /**
     * Render settings
     */
    public function render_settings() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', 'honeypot-guard-silent-anti-spam'));
        }

        $this->options = get_option('honeypot_guard_options', $this->get_default_options());
        $this->init_translator();
        $languages = HoneypotGuard_Translations::$languages;
        include HONEYPOT_GUARD_PLUGIN_DIR . 'views/settings.php';
    }

    /**
     * Render blacklist management
     */
    public function render_blacklist() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', 'honeypot-guard-silent-anti-spam'));
        }

        $this->init_translator();
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query
        $entries = $wpdb->get_results(
            "SELECT * FROM {$this->blacklist_table} ORDER BY hits DESC, created_at DESC LIMIT 100"
        );

        include HONEYPOT_GUARD_PLUGIN_DIR . 'views/blacklist.php';
    }

    /**
     * Render logs with advanced filtering
     */
    public function render_logs() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', 'honeypot-guard-silent-anti-spam'));
        }

        $this->init_translator();
        global $wpdb;

        $per_page = 20;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce not needed for pagination/filter parameters
        $current_page = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
        $offset = ($current_page - 1) * $per_page;

        // Collect all filter parameters
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Filter parameters are sanitized, no state change
        $status_filter = isset($_GET['status']) ? sanitize_key(wp_unslash($_GET['status'])) : '';
        $spam_type_filter = isset($_GET['spam_type']) ? sanitize_key(wp_unslash($_GET['spam_type'])) : '';
        $char_filter = isset($_GET['char_filter']) ? sanitize_key(wp_unslash($_GET['char_filter'])) : '';
        $form_source_filter = isset($_GET['form_source']) ? sanitize_text_field(wp_unslash($_GET['form_source'])) : '';
        $date_preset = isset($_GET['date_preset']) ? sanitize_key(wp_unslash($_GET['date_preset'])) : '';
        $date_from = isset($_GET['date_from']) ? sanitize_text_field(wp_unslash($_GET['date_from'])) : '';
        $date_to = isset($_GET['date_to']) ? sanitize_text_field(wp_unslash($_GET['date_to'])) : '';
        $specific_date = isset($_GET['specific_date']) ? sanitize_text_field(wp_unslash($_GET['specific_date'])) : '';
        $search_filter = isset($_GET['search']) ? sanitize_text_field(wp_unslash($_GET['search'])) : '';
        // phpcs:enable WordPress.Security.NonceVerification.Recommended

        // Build WHERE clauses
        $where_clauses = [];
        $where_values = [];
        $active_filters = [];

        // Status filter
        if ($status_filter) {
            $where_clauses[] = "status = %s";
            $where_values[] = $status_filter;
            $active_filters[] = ucfirst($status_filter);
        }

        // Spam type filter (searches in reason field)
        if ($spam_type_filter) {
            $spam_type_patterns = [
                'honeypot' => '%honeypot%',
                'timestamp' => '%timestamp%',
                'blacklist' => '%blacklist%',
                'cyrillic' => '%cyrillic%',
                'hanzi' => '%chinese%',
                'gibberish' => '%gibberish%',
                'links' => '%link%',
                'rate_limit' => '%rate%',
                'heuristic' => '%heuristic%',
            ];
            if (isset($spam_type_patterns[$spam_type_filter])) {
                $where_clauses[] = "reason LIKE %s";
                $where_values[] = $spam_type_patterns[$spam_type_filter];
                $active_filters[] = ucfirst($spam_type_filter);
            }
        }

        // Character filter (searches in content field)
        if ($char_filter) {
            // Cyrillic regex pattern: Unicode range \u0400-\u04FF
            // Chinese regex pattern: Unicode range \u4E00-\u9FFF
            switch ($char_filter) {
                case 'has_cyrillic':
                    $where_clauses[] = "content REGEXP '[а-яА-ЯёЁ]'";
                    $active_filters[] = 'Has Cyrillic';
                    break;
                case 'has_hanzi':
                    $where_clauses[] = "(content LIKE '%[一-龥]%' OR content REGEXP '[一-龥]' OR content REGEXP '[\x{4e00}-\x{9fff}]')";
                    $active_filters[] = 'Has Chinese';
                    break;
                case 'has_foreign':
                    $where_clauses[] = "(content REGEXP '[а-яА-ЯёЁ]' OR content REGEXP '[\x{4e00}-\x{9fff}]')";
                    $active_filters[] = 'Has Foreign Chars';
                    break;
                case 'no_cyrillic':
                    $where_clauses[] = "content NOT REGEXP '[а-яА-ЯёЁ]'";
                    $active_filters[] = 'No Cyrillic';
                    break;
                case 'no_hanzi':
                    $where_clauses[] = "content NOT REGEXP '[\x{4e00}-\x{9fff}]'";
                    $active_filters[] = 'No Chinese';
                    break;
                case 'no_foreign':
                    $where_clauses[] = "(content NOT REGEXP '[а-яА-ЯёЁ]' AND content NOT REGEXP '[\x{4e00}-\x{9fff}]')";
                    $active_filters[] = 'No Foreign Chars';
                    break;
            }
        }

        // Form source filter
        if ($form_source_filter) {
            $where_clauses[] = "form_source = %s";
            $where_values[] = $form_source_filter;
            $active_filters[] = ucfirst($form_source_filter);
        }

        // Date filters
        $today = current_time('Y-m-d');
        $date_start = '';
        $date_end = '';

        if ($specific_date) {
            $date_start = $specific_date . ' 00:00:00';
            $date_end = $specific_date . ' 23:59:59';
            $active_filters[] = 'Date: ' . $specific_date;
        } elseif ($date_preset) {
            switch ($date_preset) {
                case 'today':
                    $date_start = $today . ' 00:00:00';
                    $date_end = $today . ' 23:59:59';
                    $active_filters[] = 'Today';
                    break;
                case 'yesterday':
                    $yesterday = gmdate('Y-m-d', strtotime('-1 day', strtotime($today)));
                    $date_start = $yesterday . ' 00:00:00';
                    $date_end = $yesterday . ' 23:59:59';
                    $active_filters[] = 'Yesterday';
                    break;
                case '7days':
                    $date_start = gmdate('Y-m-d', strtotime('-7 days', strtotime($today))) . ' 00:00:00';
                    $date_end = $today . ' 23:59:59';
                    $active_filters[] = 'Last 7 Days';
                    break;
                case '30days':
                    $date_start = gmdate('Y-m-d', strtotime('-30 days', strtotime($today))) . ' 00:00:00';
                    $date_end = $today . ' 23:59:59';
                    $active_filters[] = 'Last 30 Days';
                    break;
                case 'this_month':
                    $date_start = gmdate('Y-m-01', strtotime($today)) . ' 00:00:00';
                    $date_end = $today . ' 23:59:59';
                    $active_filters[] = 'This Month';
                    break;
                case 'last_month':
                    $date_start = gmdate('Y-m-01', strtotime('first day of last month')) . ' 00:00:00';
                    $date_end = gmdate('Y-m-t', strtotime('last day of last month')) . ' 23:59:59';
                    $active_filters[] = 'Last Month';
                    break;
                case 'custom':
                    if ($date_from) {
                        $date_start = $date_from . ' 00:00:00';
                    }
                    if ($date_to) {
                        $date_end = $date_to . ' 23:59:59';
                    }
                    if ($date_from || $date_to) {
                        $active_filters[] = 'Custom Range';
                    }
                    break;
            }
        }

        if ($date_start) {
            $where_clauses[] = "created_at >= %s";
            $where_values[] = $date_start;
        }
        if ($date_end) {
            $where_clauses[] = "created_at <= %s";
            $where_values[] = $date_end;
        }

        // Search filter
        if ($search_filter) {
            $search_like = '%' . $wpdb->esc_like($search_filter) . '%';
            $where_clauses[] = "(content LIKE %s OR reason LIKE %s OR ip_address LIKE %s OR source_url LIKE %s)";
            $where_values[] = $search_like;
            $where_values[] = $search_like;
            $where_values[] = $search_like;
            $where_values[] = $search_like;
            $active_filters[] = 'Search: ' . $search_filter;
        }

        // Build final WHERE clause
        $where = '';
        if (!empty($where_clauses)) {
            $where = ' WHERE ' . implode(' AND ', $where_clauses);
        }

        // Get total count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table name
        $count_sql = "SELECT COUNT(*) FROM {$this->table_name}" . $where;
        if (!empty($where_values)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $count_sql uses placeholders from $where_clauses
            $count_sql = $wpdb->prepare($count_sql, $where_values);
        }
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query is safe when no user-controlled values in $where
        $total_items = (int) $wpdb->get_var($count_sql);
        $total_pages = ceil($total_items / $per_page);

        // Get logs
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table name, $where contains prepared placeholders
        $sql = "SELECT * FROM {$this->table_name} {$where} ORDER BY created_at DESC LIMIT %d OFFSET %d";
        $query_values = array_merge($where_values, [$per_page, $offset]);
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- $sql is prepared with $query_values which includes all user inputs
        $logs = $wpdb->get_results($wpdb->prepare($sql, $query_values));

        // Get unique form sources for filter dropdown
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query
        $form_sources = $wpdb->get_col("SELECT DISTINCT form_source FROM {$this->table_name} WHERE form_source IS NOT NULL AND form_source != '' ORDER BY form_source");

        // Get all blacklisted IPs for button state
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table query
        $blacklisted_ips = $wpdb->get_col("SELECT value FROM {$this->blacklist_table} WHERE type = 'ip'");
        $blacklisted_ips = array_map('strtolower', $blacklisted_ips);

        // Check if any filters are active
        $has_filters = !empty($active_filters);

        include HONEYPOT_GUARD_PLUGIN_DIR . 'views/logs.php';
    }

    /**
     * =========================================================================
     * AJAX HANDLERS
     * =========================================================================
     */

    /**
     * AJAX: Delete log
     */
    public function ajax_delete_log() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        $log_id = absint($_POST['log_id'] ?? 0);
        if (!$log_id) {
            wp_send_json_error(['message' => 'Invalid ID'], 400);
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table delete by ID
        $wpdb->delete($this->table_name, ['id' => $log_id], ['%d']);

        wp_send_json_success();
    }

    /**
     * AJAX: Clear logs
     */
    public function ajax_clear_logs() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table truncate
        $wpdb->query("TRUNCATE TABLE {$this->table_name}");

        wp_send_json_success();
    }

    /**
     * AJAX: Get stats
     */
    public function ajax_get_stats() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        global $wpdb;

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table stats
        wp_send_json_success([
            'total_spam' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam'"),
            'total_clean' => (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name} WHERE status='clean'"),
            'spam_today' => (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name} WHERE status='spam' AND created_at >= %s",
                gmdate('Y-m-d 00:00:00')
            ))
        ]);
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    }

    /**
     * AJAX: Add blacklist entry
     */
    public function ajax_add_blacklist() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        $type = isset($_POST['type']) ? sanitize_key(wp_unslash($_POST['type'])) : '';
        $value = isset($_POST['value']) ? sanitize_text_field(wp_unslash($_POST['value'])) : '';
        $reason = isset($_POST['reason']) ? sanitize_text_field(wp_unslash($_POST['reason'])) : '';

        if (!in_array($type, ['email', 'domain', 'keyword', 'ip'])) {
            wp_send_json_error(['message' => 'Invalid type'], 400);
        }

        if (empty($value)) {
            wp_send_json_error(['message' => 'Value required'], 400);
        }

        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert
        $result = $wpdb->insert(
            $this->blacklist_table,
            [
                'type' => $type,
                'value' => strtolower($value),
                'reason' => $reason,
                'created_by' => get_current_user_id(),
                'created_at' => current_time('mysql')
            ],
            ['%s', '%s', '%s', '%d', '%s']
        );

        if ($result === false) {
            wp_send_json_error(['message' => 'Entry may already exist'], 400);
        }

        // Clear blacklist cache
        $this->clear_blacklist_cache();

        wp_send_json_success(['id' => $wpdb->insert_id]);
    }

    /**
     * AJAX: Remove blacklist entry
     */
    public function ajax_remove_blacklist() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        $id = absint($_POST['id'] ?? 0);
        if (!$id) {
            wp_send_json_error(['message' => 'Invalid ID'], 400);
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table delete by ID
        $wpdb->delete($this->blacklist_table, ['id' => $id], ['%d']);

        // Clear blacklist cache
        $this->clear_blacklist_cache();

        wp_send_json_success();
    }

    /**
     * AJAX: Remove blacklist entry by value
     */
    public function ajax_remove_blacklist_by_value() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        $type = isset($_POST['type']) ? sanitize_key(wp_unslash($_POST['type'])) : '';
        $value = isset($_POST['value']) ? sanitize_text_field(wp_unslash($_POST['value'])) : '';

        if (!in_array($type, ['email', 'domain', 'keyword', 'ip'])) {
            wp_send_json_error(['message' => 'Invalid type'], 400);
        }

        if (empty($value)) {
            wp_send_json_error(['message' => 'Value required'], 400);
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table, delete by value
        $result = $wpdb->delete(
            $this->blacklist_table,
            ['type' => $type, 'value' => strtolower($value)],
            ['%s', '%s']
        );

        // Clear blacklist cache
        $this->clear_blacklist_cache();

        if ($result === false) {
            wp_send_json_error(['message' => 'Failed to remove entry'], 500);
        }

        wp_send_json_success(['removed' => $result]);
    }

    /**
     * AJAX: Test submission
     */
    public function ajax_test_submission() {
        check_ajax_referer('honeypot_guard_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized'], 403);
        }

        $test_data = [
            'email' => isset($_POST['test_email']) ? sanitize_email(wp_unslash($_POST['test_email'])) : '',
            'subject' => isset($_POST['test_subject']) ? sanitize_text_field(wp_unslash($_POST['test_subject'])) : '',
            'message' => isset($_POST['test_message']) ? sanitize_textarea_field(wp_unslash($_POST['test_message'])) : ''
        ];

        // Temporarily disable logging for test
        $original_log = $this->options['log_clean_submissions'];
        $this->options['log_clean_submissions'] = false;

        $result = $this->analyze_submission($test_data, 'test');

        $this->options['log_clean_submissions'] = $original_log;

        wp_send_json_success($result);
    }

    /**
     * Cleanup old logs
     */
    public function cleanup_old_logs() {
        global $wpdb;

        $days = absint($this->options['log_retention_days']);
        if ($days < 1) $days = 30;

        $cutoff = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table cleanup
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE created_at < %s",
            $cutoff
        ));
    }
}

// Initialize
HoneypotGuard::get_instance();
