/**
 * Honeypot Guard Admin JavaScript
 * Consolidated admin functionality for all plugin pages
 */

(function($) {
    'use strict';

    var HG_Admin = {
        init: function() {
            this.bindEvents();
            this.initSettingsPage();
            this.initLogsPage();
            this.initBlacklistPage();
        },

        bindEvents: function() {
            // Delete single log
            $(document).on('click', '.oh-delete-log', this.deleteLog);

            // Clear all logs - use delegated binding for reliability
            $(document).on('click', '#oh-clear-logs', this.clearLogs);

            // Export logs - use delegated binding for reliability
            $(document).on('click', '#oh-export-logs', this.exportLogs);

            // View content modal
            $(document).on('click', '.oh-view-content', this.showContentModal);

            // Conditional fields (for settings page)
            this.initConditionalFields();
            $(document).on('change', '#spam_message_mode', this.handleConditionalFields);

            // Close modal
            $(document).on('click', '.oh-modal-close', this.closeModal);
            $(document).on('click', '.oh-modal', function(e) {
                if ($(e.target).hasClass('oh-modal')) {
                    HG_Admin.closeModal();
                }
            });

            // Keyboard escape to close modal
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape') {
                    HG_Admin.closeModal();
                }
            });
        },

        // =============================================
        // Settings Page Functionality
        // =============================================
        initSettingsPage: function() {
            // Only run on settings page
            if (!$('#honeypot_mode').length) return;

            // Honeypot mode toggle
            $('#honeypot_mode').on('change', function() {
                var mode = $(this).val();
                $('#custom-field-name').toggle(mode === 'custom');
                $('#rotation-setting').toggle(mode === 'dynamic');
            });

            // AI gibberish toggle
            $('#gibberish_enabled').on('change', function() {
                $('#ai-settings-wrapper').toggle(this.checked);
            });

            // Spam message mode toggle
            $('#spam_message_mode').on('change', function() {
                var mode = $(this).val();
                $('.oh-conditional-field').each(function() {
                    var $field = $(this);
                    var showWhen = $field.data('show-when');
                    var showValue = $field.data('show-value');
                    if (showWhen === 'spam_message_mode') {
                        $field.toggle(mode === showValue);
                    }
                });
            }).trigger('change');

            // Card collapse/expand functionality with localStorage persistence
            this.initCardStates();
        },

        initCardStates: function() {
            var STORAGE_KEY = 'oh_card_states';

            function getCardStates() {
                try {
                    var states = localStorage.getItem(STORAGE_KEY);
                    return states ? JSON.parse(states) : {};
                } catch (e) {
                    return {};
                }
            }

            function saveCardState(cardId, isCollapsed) {
                try {
                    var states = getCardStates();
                    states[cardId] = isCollapsed;
                    localStorage.setItem(STORAGE_KEY, JSON.stringify(states));
                } catch (e) {
                    // localStorage not available
                }
            }

            function toggleCard($card) {
                var cardId = $card.data('card-id');
                var isCurrentlyCollapsed = $card.hasClass('oh-collapsed');

                if (isCurrentlyCollapsed) {
                    $card.removeClass('oh-collapsed');
                    $card.find('.oh-card-body').first().show();
                } else {
                    $card.addClass('oh-collapsed');
                    $card.find('.oh-card-body').first().hide();
                }

                saveCardState(cardId, !isCurrentlyCollapsed);
            }

            // Initialize card states from localStorage
            var states = getCardStates();
            $('.oh-card[data-card-id]').each(function() {
                var $card = $(this);
                var cardId = $card.data('card-id');
                // Default: all open (not collapsed)
                if (states[cardId] === true) {
                    $card.addClass('oh-collapsed');
                    $card.find('.oh-card-body').first().hide();
                }
            });

            // Toggle card on header click - using delegation for better compatibility
            $(document).on('click', '.oh-card-header.oh-collapsible', function(e) {
                // Don't toggle if clicking on help button or tooltip
                var $target = $(e.target);
                if ($target.closest('.oh-help-btn-wrap').length) {
                    return;
                }

                e.preventDefault();
                var $card = $(this).closest('.oh-card');
                if ($card.length && $card.data('card-id')) {
                    toggleCard($card);
                }
            });

            // Help button - prevent card toggle when clicking help area
            $(document).on('click', '.oh-help-btn-wrap, .oh-help-btn', function(e) {
                e.stopPropagation();
            });
        },

        // =============================================
        // Logs Page Functionality
        // =============================================
        initLogsPage: function() {
            // Only run on logs page
            if (!$('#oh-toggle-filters').length) return;

            var self = this;

            // Toggle filters panel
            $('#oh-toggle-filters').on('click', function() {
                var panel = $('#oh-filters-panel');
                panel.slideToggle(200);
                var strings = typeof honeypotGuardAdmin !== 'undefined' ? honeypotGuardAdmin.strings : {};
                $(this).text(panel.is(':visible') ? (strings.hideFilters || 'Hide Filters') : (strings.showFilters || 'Show Filters'));
            });

            // Add IP to blacklist
            $(document).on('click', '.oh-blacklist-ip', function() {
                var $btn = $(this);
                var ip = $btn.data('ip');
                var strings = honeypotGuardAdmin.strings;

                if (!confirm((strings.confirmBlacklistIp || 'Block IP') + ' ' + ip + '?')) {
                    return;
                }

                $btn.prop('disabled', true);

                $.ajax({
                    url: honeypotGuardAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'honeypot_guard_add_blacklist',
                        nonce: honeypotGuardAdmin.nonce,
                        type: 'ip',
                        value: ip,
                        reason: strings.addedFromSpamLogs || 'Added from spam logs'
                    },
                    success: function(response) {
                        if (response.success) {
                            $btn.removeClass('oh-blacklist-ip')
                                .addClass('oh-unblacklist-ip')
                                .prop('disabled', false)
                                .attr('title', strings.removeIpFromBlacklist || 'Remove from blacklist')
                                .find('.dashicons')
                                .removeClass('dashicons-dismiss')
                                .addClass('dashicons-yes-alt');
                        } else {
                            alert(response.data.message || strings.error);
                            $btn.prop('disabled', false);
                        }
                    },
                    error: function() {
                        alert(strings.error);
                        $btn.prop('disabled', false);
                    }
                });
            });

            // Remove IP from blacklist
            $(document).on('click', '.oh-unblacklist-ip', function() {
                var $btn = $(this);
                var ip = $btn.data('ip');
                var strings = honeypotGuardAdmin.strings;

                if (!confirm((strings.confirmUnblacklistIp || 'Unblock IP') + ' ' + ip + '?')) {
                    return;
                }

                $btn.prop('disabled', true);

                $.ajax({
                    url: honeypotGuardAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'honeypot_guard_remove_blacklist_by_value',
                        nonce: honeypotGuardAdmin.nonce,
                        type: 'ip',
                        value: ip
                    },
                    success: function(response) {
                        if (response.success) {
                            $btn.removeClass('oh-unblacklist-ip')
                                .addClass('oh-blacklist-ip')
                                .prop('disabled', false)
                                .attr('title', strings.addIpToBlacklist || 'Add to blacklist')
                                .find('.dashicons')
                                .removeClass('dashicons-yes-alt')
                                .addClass('dashicons-dismiss');
                        } else {
                            alert(response.data.message || strings.error);
                            $btn.prop('disabled', false);
                        }
                    },
                    error: function() {
                        alert(strings.error);
                        $btn.prop('disabled', false);
                    }
                });
            });

            // Show/hide custom date fields based on preset selection
            $('#oh-date-preset').on('change', function() {
                var val = $(this).val();
                if (val === 'custom') {
                    $('.oh-custom-date').show();
                } else {
                    $('.oh-custom-date').hide();
                    $('input[name="date_from"], input[name="date_to"]').val('');
                }
                // Clear specific date when using preset
                if (val) {
                    $('input[name="specific_date"]').val('');
                }
            });

            // Clear preset when using specific date
            $('input[name="specific_date"]').on('change', function() {
                if ($(this).val()) {
                    $('#oh-date-preset').val('');
                    $('.oh-custom-date').hide();
                }
            });
        },

        // =============================================
        // Blacklist Page Functionality
        // =============================================
        initBlacklistPage: function() {
            // Only run on blacklist page
            if (!$('#oh-add-blacklist').length) return;

            var strings = typeof honeypotGuardAdmin !== 'undefined' ? honeypotGuardAdmin.strings : {};

            // Add blacklist entry
            $('#oh-add-blacklist').on('click', function() {
                var type = $('#oh-bl-type').val();
                var value = $('#oh-bl-value').val().trim();
                var reason = $('#oh-bl-reason').val().trim();

                if (!value) {
                    alert(strings.enterValueToBlock || 'Please enter a value to block');
                    return;
                }

                var $btn = $(this);
                $btn.prop('disabled', true);

                $.ajax({
                    url: honeypotGuardAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'honeypot_guard_add_blacklist',
                        nonce: honeypotGuardAdmin.nonce,
                        type: type,
                        value: value,
                        reason: reason
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || strings.error);
                        }
                    },
                    error: function() {
                        alert(strings.error);
                    },
                    complete: function() {
                        $btn.prop('disabled', false);
                    }
                });
            });

            // Remove blacklist entry
            $('.oh-remove-blacklist').on('click', function() {
                if (!confirm(strings.confirmDelete || 'Are you sure?')) {
                    return;
                }

                var $row = $(this).closest('tr');
                var id = $row.data('id');

                $.ajax({
                    url: honeypotGuardAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'honeypot_guard_remove_blacklist',
                        nonce: honeypotGuardAdmin.nonce,
                        id: id
                    },
                    success: function(response) {
                        if (response.success) {
                            $row.fadeOut(300, function() { $(this).remove(); });
                        } else {
                            alert(strings.error);
                        }
                    },
                    error: function() {
                        alert(strings.error);
                    }
                });
            });

            // Enter key to add
            $('#oh-bl-value, #oh-bl-reason').on('keypress', function(e) {
                if (e.which === 13) {
                    e.preventDefault();
                    $('#oh-add-blacklist').click();
                }
            });
        },

        // =============================================
        // Common Functionality
        // =============================================
        deleteLog: function(e) {
            e.preventDefault();

            if (!confirm(honeypotGuardAdmin.strings.confirmDelete)) {
                return;
            }

            var $row = $(this).closest('tr');
            var logId = $row.data('log-id');

            $.ajax({
                url: honeypotGuardAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'honeypot_guard_delete_log',
                    nonce: honeypotGuardAdmin.nonce,
                    log_id: logId
                },
                beforeSend: function() {
                    $row.css('opacity', '0.5');
                },
                success: function(response) {
                    if (response.success) {
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            HG_Admin.showNotice(honeypotGuardAdmin.strings.deleted, 'success');
                        });
                    } else {
                        $row.css('opacity', '1');
                        HG_Admin.showNotice(honeypotGuardAdmin.strings.error, 'error');
                    }
                },
                error: function() {
                    $row.css('opacity', '1');
                    HG_Admin.showNotice(honeypotGuardAdmin.strings.error, 'error');
                }
            });
        },

        clearLogs: function(e) {
            e.preventDefault();

            // Check if honeypotGuardAdmin exists
            if (typeof honeypotGuardAdmin === 'undefined') {
                alert('Error: Admin settings not loaded. Please refresh the page.');
                return;
            }

            if (!confirm(honeypotGuardAdmin.strings.confirmClear || 'Are you sure you want to clear all logs?')) {
                return;
            }

            var $btn = $(this);

            $.ajax({
                url: honeypotGuardAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'honeypot_guard_clear_logs',
                    nonce: honeypotGuardAdmin.nonce
                },
                beforeSend: function() {
                    $btn.prop('disabled', true).text('Clearing...');
                },
                success: function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        var errorMsg = response.data && response.data.message ? response.data.message : (honeypotGuardAdmin.strings.error || 'An error occurred');
                        HG_Admin.showNotice(errorMsg, 'error');
                        $btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Clear All Logs');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('HoneypotGuard: Clear logs failed', status, error);
                    HG_Admin.showNotice(honeypotGuardAdmin.strings.error || 'An error occurred', 'error');
                    $btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Clear All Logs');
                }
            });
        },

        exportLogs: function(e) {
            e.preventDefault();

            var rows = [];
            var headers = ['Status', 'Score', 'Reason', 'Source', 'Page URL', 'IP', 'Time'];
            rows.push(headers.join(','));

            $('.oh-table-logs tbody tr').each(function() {
                var $row = $(this);
                var pageUrl = $row.find('.oh-url a').attr('href') || $row.find('.oh-url').text().trim() || '';
                var row = [
                    $row.find('.oh-status').text().trim(),
                    $row.find('.oh-score').text().trim(),
                    '"' + $row.find('.oh-reason').text().trim().replace(/"/g, '""') + '"',
                    $row.find('td').eq(4).text().trim(),
                    '"' + pageUrl.replace(/"/g, '""') + '"',
                    $row.find('.oh-ip').text().trim(),
                    $row.find('.oh-time span').attr('title') || $row.find('.oh-time').text().trim()
                ];
                rows.push(row.join(','));
            });

            var csvContent = rows.join('\n');
            var blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
            var link = document.createElement('a');
            var url = URL.createObjectURL(blob);

            link.setAttribute('href', url);
            link.setAttribute('download', 'honeypot-guard-logs-' + HG_Admin.formatDate(new Date()) + '.csv');
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        },

        showContentModal: function(e) {
            e.preventDefault();

            var $btn = $(this);
            var displayHtml = '';

            // Get base64 encoded content from data attribute
            var base64Content = $btn.attr('data-content') || '';

            if (!base64Content) {
                displayHtml = '<p class="oh-empty" style="padding:20px;text-align:center;color:#666;">No content available</p>';
                $('#oh-modal-content-display').html(displayHtml);
                $('#oh-content-modal').fadeIn(200);
                return;
            }

            try {
                // Decode base64 to JSON string, then parse JSON
                var jsonContent = atob(base64Content);
                var data = JSON.parse(jsonContent);

                if (typeof data === 'object' && data !== null) {
                    // Check if it's wrapped raw data
                    if (data._raw !== undefined) {
                        displayHtml = '<div class="oh-content-text" style="padding:10px;white-space:pre-wrap;word-break:break-word;">' + HG_Admin.escapeHtml(String(data._raw)) + '</div>';
                    } else if (Object.keys(data).length === 0) {
                        displayHtml = '<p class="oh-empty" style="padding:20px;text-align:center;color:#666;">No form data available</p>';
                    } else {
                        displayHtml = HG_Admin.formatFormData(data);
                    }
                } else {
                    displayHtml = '<div class="oh-content-text" style="padding:10px;white-space:pre-wrap;word-break:break-word;">' + HG_Admin.escapeHtml(String(data)) + '</div>';
                }
            } catch (err) {
                console.error('HoneypotGuard: Error decoding content:', err);
                // Try to show something useful
                try {
                    var rawContent = atob(base64Content);
                    displayHtml = '<div class="oh-content-text" style="padding:10px;white-space:pre-wrap;word-break:break-word;background:#f5f5f5;border-radius:4px;">' + HG_Admin.escapeHtml(rawContent) + '</div>';
                } catch (e2) {
                    displayHtml = '<p class="oh-empty" style="padding:20px;text-align:center;color:#666;">Error loading content</p>';
                }
            }

            // Ensure we always have something to display
            if (!displayHtml) {
                displayHtml = '<p class="oh-empty" style="padding:20px;text-align:center;color:#666;">No content to display</p>';
            }

            $('#oh-modal-content-display').html(displayHtml);
            $('#oh-content-modal').fadeIn(200);
        },

        formatFormData: function(data) {
            var html = '<table class="oh-content-table" style="width:100%;border-collapse:collapse;"><tbody>';
            var rowCount = 0;

            for (var key in data) {
                if (data.hasOwnProperty(key)) {
                    // Skip internal fields (honeypot, CF7, WP internal)
                    if (key.indexOf('_oh_') === 0 ||
                        key.indexOf('_wpcf7') === 0 ||
                        key.indexOf('_wp') === 0 ||
                        key === '_raw') {
                        continue;
                    }

                    var value = data[key];
                    var displayKey = HG_Admin.formatFieldName(key);
                    var displayValue = '';

                    if (value === null || value === undefined) {
                        displayValue = '';
                    } else if (Array.isArray(value)) {
                        displayValue = value.join(', ');
                    } else if (typeof value === 'object') {
                        displayValue = JSON.stringify(value);
                    } else {
                        displayValue = String(value);
                    }

                    html += '<tr style="border-bottom:1px solid #eee;">';
                    html += '<th style="padding:8px 12px;text-align:left;background:#f8f9fa;font-weight:600;width:30%;">' + HG_Admin.escapeHtml(displayKey) + '</th>';
                    html += '<td style="padding:8px 12px;word-break:break-word;">' + HG_Admin.escapeHtml(displayValue) + '</td>';
                    html += '</tr>';
                    rowCount++;
                }
            }

            html += '</tbody></table>';

            if (rowCount === 0) {
                return '<p class="oh-empty" style="padding:20px;text-align:center;color:#666;">No form data available</p>';
            }

            return html;
        },

        formatFieldName: function(key) {
            // Convert field names like "your-name" or "your_email" to "Your Name" or "Your Email"
            return key
                .replace(/[-_]/g, ' ')
                .replace(/\b\w/g, function(l) { return l.toUpperCase(); });
        },

        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        initConditionalFields: function() {
            // Initialize conditional field visibility on page load
            HG_Admin.handleConditionalFields();
        },

        handleConditionalFields: function() {
            var $select = $('#spam_message_mode');
            if (!$select.length) return;

            var selectedValue = $select.val();

            // Hide all conditional fields first
            $('.oh-conditional-field').hide();

            // Show fields that match the selected value
            $('.oh-conditional-field').each(function() {
                var $field = $(this);
                var showWhen = $field.data('show-when');
                var showValue = $field.data('show-value');

                if (showWhen === 'spam_message_mode' && showValue === selectedValue) {
                    $field.show();
                }
            });
        },

        closeModal: function() {
            $('.oh-modal').fadeOut(200);
        },

        showNotice: function(message, type) {
            var $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');
            $('.honeypot-guard-wrap .oh-header, .honeypot-guard-wrap .oh-header').after($notice);

            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 3000);
        },

        formatDate: function(date) {
            var year = date.getFullYear();
            var month = String(date.getMonth() + 1).padStart(2, '0');
            var day = String(date.getDate()).padStart(2, '0');
            return year + '-' + month + '-' + day;
        }
    };

    $(document).ready(function() {
        HG_Admin.init();
    });

})(jQuery);
