/**
 * Open Honeypot Anti-Spam Suite - Standalone Version
 * Version: 2.0.1
 *
 * Universal honeypot and anti-spam protection for any HTML website.
 * Simply include this script on your page to protect all forms.
 *
 * Usage:
 * <script src="open-honeypot.js"></script>
 * <script>
 *   OpenHoneypot.configure({
 *     honeypotFieldName: 'middle_name',
 *     minSubmissionTime: 2,
 *     // ... other options
 *   });
 * </script>
 *
 * Learn more: https://github.com/user/open-honeypot
 * Professional support: https://www.mantasdigital.com/
 */

(function(root, factory) {
    'use strict';

    if (typeof module === 'object' && module.exports) {
        module.exports = factory();
    } else if (typeof define === 'function' && define.amd) {
        define(factory);
    } else {
        root.OpenHoneypot = factory();
    }
}(typeof self !== 'undefined' ? self : this, function() {
    'use strict';

    // =========================================================================
    // DEFAULT CONFIGURATION
    // =========================================================================

    var defaultConfig = {
        // Honeypot Settings
        honeypotEnabled: true,
        honeypotFieldName: 'extra_billing_note',
        autoGenerateFieldName: true,
        multipleHoneypots: false,
        additionalHoneypotNames: ['middle_name', 'fax_number'],

        // Timestamp Settings
        timestampEnabled: true,
        minSubmissionTime: 2,
        maxSubmissionTime: 3600, // 1 hour

        // Cookie Verification
        cookieCheckEnabled: false,

        // Referer Check
        requireReferer: false,

        // Rate Limiting
        rateLimitEnabled: false,
        rateLimitSubmissions: 5,
        rateLimitWindow: 60, // seconds

        // Character Filtering
        blockCyrillic: false,
        blockHanzi: false,
        blockBengali: false, // South Asian scripts
        blockLatin: false, // Use with caution!

        // Keyword Blacklist (comma-separated string or array)
        keywordBlacklist: [],

        // Email Blacklist (comma-separated emails or patterns like *@spam.com)
        emailBlacklist: '',

        // Domain Blacklist (comma-separated domains found in URLs/content)
        domainBlacklist: '',

        // Gibberish Detection (keyboard mashing, random text)
        gibberishDetectionEnabled: true,

        // Heuristic Analysis
        heuristicEnabled: true,
        maxLinks: 3,

        // Spam Threshold (0-100)
        spamThreshold: 70,

        // Form Selectors
        formSelectors: ['form'],
        excludeSelectors: ['.oh-exclude', '[data-oh-exclude]'],

        // Callbacks
        onSpamDetected: null, // function(form, result) {}
        onValidSubmission: null, // function(form, result) {}

        // Behavior
        blockSubmission: true, // Set to false to only log/callback without blocking
        showErrorMessage: true,
        errorMessage: 'Your submission could not be processed. Please try again.',

        // Escalation
        escalationEnabled: true,
        escalationThreshold: 3,
        escalationWindow: 60, // minutes
        supportUrl: 'https://www.mantasdigital.com/',

        // Debug
        debug: false
    };

    // Realistic honeypot field names
    var honeypotNames = [
        'middle_name',
        'user_verify_id',
        'extra_billing_note',
        'secondary_phone',
        'alt_email_address',
        'company_extension',
        'fax_number',
        'preferred_contact_time',
        'referral_source_detail',
        'additional_notes_field'
    ];

    // =========================================================================
    // OPEN HONEYPOT CORE
    // =========================================================================

    var OpenHoneypot = {
        version: '2.0.1',
        config: Object.assign({}, defaultConfig),
        initialized: false,
        csrfToken: null,
        verifyCookie: null,
        spamTimestamps: [],
        submissionTimestamps: [],

        /**
         * Configure OpenHoneypot options
         */
        configure: function(options) {
            if (options && typeof options === 'object') {
                // Handle keyword blacklist
                if (typeof options.keywordBlacklist === 'string') {
                    options.keywordBlacklist = options.keywordBlacklist.split(',').map(function(k) {
                        return k.trim().toLowerCase();
                    }).filter(Boolean);
                }

                Object.assign(this.config, options);
            }

            this.log('Configuration updated:', this.config);
            return this;
        },

        /**
         * Initialize OpenHoneypot
         */
        init: function(options) {
            var self = this;

            if (this.initialized) {
                this.log('Already initialized, updating config');
                return this.configure(options);
            }

            this.configure(options);
            this.csrfToken = this.generateToken();
            this.loadSpamCount();
            this.loadSubmissionTimestamps();

            // Set verification cookie if enabled
            if (this.config.cookieCheckEnabled) {
                this.setVerifyCookie();
            }

            // Auto-generate honeypot field name if enabled
            if (this.config.autoGenerateFieldName) {
                var hash = this.simpleHash(window.location.hostname);
                this.config.honeypotFieldName = honeypotNames[hash % honeypotNames.length];
            }

            // Wait for DOM ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', function() {
                    self.setup();
                });
            } else {
                this.setup();
            }

            this.initialized = true;
            this.log('Open Honeypot initialized, version ' + this.version);

            return this;
        },

        /**
         * Setup form protection
         */
        setup: function() {
            var self = this;

            // Find and protect forms
            this.config.formSelectors.forEach(function(selector) {
                var forms = document.querySelectorAll(selector);
                forms.forEach(function(form) {
                    if (!self.shouldExcludeForm(form)) {
                        self.protect(form);
                    }
                });
            });

            // Check escalation
            this.checkEscalation();

            // Observe for dynamically added forms
            this.observeDOM();
        },

        /**
         * Protect a specific form
         */
        protect: function(form) {
            var self = this;

            if (!(form instanceof HTMLFormElement)) {
                form = document.querySelector(form);
            }

            if (!form || form.hasAttribute('data-oh-protected')) {
                return this;
            }

            form.setAttribute('data-oh-protected', 'true');

            // Inject protective fields
            if (this.config.honeypotEnabled) {
                this.injectHoneypot(form);

                if (this.config.multipleHoneypots) {
                    this.injectMultipleHoneypots(form);
                }
            }

            if (this.config.timestampEnabled) {
                this.injectTimestamp(form);
            }

            this.injectCSRFToken(form);

            // Intercept submission
            form.addEventListener('submit', function(e) {
                var result = self.validate(form);

                if (result.isSpam) {
                    if (self.config.blockSubmission) {
                        e.preventDefault();
                        e.stopPropagation();
                    }

                    self.handleSpam(form, result);

                    if (self.config.onSpamDetected) {
                        self.config.onSpamDetected(form, result);
                    }

                    return false;
                }

                // Track submission for rate limiting
                if (self.config.rateLimitEnabled) {
                    self.trackSubmission();
                }

                if (self.config.onValidSubmission) {
                    self.config.onValidSubmission(form, result);
                }
            });

            this.log('Form protected:', form);

            return this;
        },

        /**
         * Manually validate form data
         */
        validate: function(formOrData) {
            var result = {
                isSpam: false,
                score: 0,
                reasons: [],
                details: {}
            };

            var formData;
            var form = null;

            if (formOrData instanceof HTMLFormElement) {
                form = formOrData;
                formData = new FormData(form);
            } else if (formOrData instanceof FormData) {
                formData = formOrData;
            } else if (typeof formOrData === 'object') {
                formData = new FormData();
                for (var key in formOrData) {
                    if (formOrData.hasOwnProperty(key)) {
                        formData.append(key, formOrData[key]);
                    }
                }
            } else {
                this.log('Invalid input for validation');
                return result;
            }

            var content = this.extractContent(formData);

            // 1. Rate Limit Check
            if (this.config.rateLimitEnabled && this.isRateLimited()) {
                result.score += 100;
                result.reasons.push('Rate limit exceeded');
                result.details.rateLimited = true;
            }

            // 2. Referer Check
            if (this.config.requireReferer && !document.referrer) {
                result.score += 30;
                result.reasons.push('Missing referer');
                result.details.noReferer = true;
            }

            // 3. Cookie Verification Check
            if (this.config.cookieCheckEnabled) {
                var cookie = this.getCookie('_oh_verify');
                if (!cookie) {
                    result.score += 40;
                    result.reasons.push('Missing verification cookie');
                    result.details.noCookie = true;
                }
            }

            // 4. Honeypot Check
            if (this.config.honeypotEnabled) {
                var honeypotValue = formData.get(this.config.honeypotFieldName);
                if (honeypotValue && honeypotValue.toString().trim() !== '') {
                    result.score += 100;
                    result.reasons.push('Honeypot triggered');
                    result.details.honeypot = true;
                }

                // Check multiple honeypots
                if (this.config.multipleHoneypots) {
                    var additionalNames = this.config.additionalHoneypotNames || [];
                    for (var i = 0; i < additionalNames.length; i++) {
                        var val = formData.get(additionalNames[i]);
                        if (val && val.toString().trim() !== '') {
                            result.score += 100;
                            result.reasons.push('Additional honeypot filled: ' + additionalNames[i]);
                            result.details.multipleHoneypot = true;
                            break;
                        }
                    }
                }
            }

            // 5. Timestamp Check
            if (this.config.timestampEnabled) {
                var timestamp = parseInt(formData.get('_oh_ts'), 10);
                if (timestamp) {
                    var elapsed = Math.floor(Date.now() / 1000) - timestamp;
                    if (elapsed < this.config.minSubmissionTime) {
                        result.score += 80;
                        result.reasons.push('Submitted too fast (' + elapsed + 's)');
                        result.details.tooFast = true;
                    }
                    if (elapsed > this.config.maxSubmissionTime) {
                        result.score += 40;
                        result.reasons.push('Form session expired');
                        result.details.expired = true;
                    }
                }
            }

            // 6. CSRF Check
            var submittedToken = formData.get('_oh_tk');
            if (this.csrfToken && submittedToken !== this.csrfToken) {
                result.score += 50;
                result.reasons.push('Invalid security token');
                result.details.invalidToken = true;
            }

            // 7. Email Blacklist Check
            if (this.config.emailBlacklist) {
                var emailResult = this.checkEmailBlacklist(formData);
                if (emailResult.matched) {
                    result.score += 100;
                    result.reasons.push('Email blacklisted: ' + emailResult.email);
                    result.details.emailBlacklisted = emailResult.email;
                }
            }

            // 8. Domain Blacklist Check
            if (this.config.domainBlacklist) {
                var domainResult = this.checkDomainBlacklist(content, formData);
                if (domainResult.matched) {
                    result.score += 90;
                    result.reasons.push('Domain blacklisted: ' + domainResult.domain);
                    result.details.domainBlacklisted = domainResult.domain;
                }
            }

            // 9. Character Filtering
            // Cyrillic
            if (this.config.blockCyrillic && /[\u0400-\u04FF]/.test(content)) {
                result.score += 60;
                result.reasons.push('Contains Cyrillic characters');
                result.details.cyrillic = true;
            }

            // Chinese
            if (this.config.blockHanzi && /[\u4E00-\u9FFF]/.test(content)) {
                result.score += 60;
                result.reasons.push('Contains Chinese characters');
                result.details.hanzi = true;
            }

            // South Asian Scripts (Bengali, Devanagari, Tamil, Telugu, etc.)
            if (this.config.blockBengali && /[\u0980-\u09FF\u0900-\u097F\u0B80-\u0BFF\u0C00-\u0C7F\u0C80-\u0CFF\u0D00-\u0D7F\u0A80-\u0AFF\u0B00-\u0B7F\u0A00-\u0A7F]/.test(content)) {
                result.score += 60;
                result.reasons.push('Contains South Asian script characters');
                result.details.bengali = true;
            }

            // Latin (use with caution)
            if (this.config.blockLatin && /[a-zA-Z]/.test(content)) {
                result.score += 60;
                result.reasons.push('Contains Latin characters');
                result.details.latin = true;
            }

            // 10. Keyword Blacklist
            if (this.config.keywordBlacklist && this.config.keywordBlacklist.length > 0) {
                var contentLower = content.toLowerCase();
                for (var j = 0; j < this.config.keywordBlacklist.length; j++) {
                    var keyword = this.config.keywordBlacklist[j];
                    if (keyword && contentLower.indexOf(keyword) !== -1) {
                        result.score += 90;
                        result.reasons.push('Blacklisted keyword: ' + keyword);
                        result.details.blacklistMatch = keyword;
                        break;
                    }
                }
            }

            // 11. Gibberish Detection
            if (this.config.gibberishDetectionEnabled) {
                var gibberishResult = this.checkGibberish(formData);
                if (gibberishResult.score > 0) {
                    result.score += gibberishResult.score;
                    result.reasons = result.reasons.concat(gibberishResult.reasons);
                    Object.assign(result.details, gibberishResult.details);
                }
            }

            // 12. Heuristic Analysis
            if (this.config.heuristicEnabled) {
                var heuristic = this.runHeuristics(content);
                result.score += heuristic.score;
                result.reasons = result.reasons.concat(heuristic.reasons);
                Object.assign(result.details, heuristic.details);
            }

            // Final determination
            result.isSpam = result.score >= this.config.spamThreshold;

            this.log('Validation result:', result);

            return result;
        },

        /**
         * Email Blacklist Check
         */
        checkEmailBlacklist: function(formData) {
            var blacklistStr = typeof this.config.emailBlacklist === 'string' ? this.config.emailBlacklist : '';
            var blacklist = blacklistStr.toLowerCase().split(',').map(function(e) {
                return e.trim();
            }).filter(Boolean);

            if (blacklist.length === 0) return { matched: false };

            // Find email fields
            var emailFields = ['email', 'your-email', 'mail', 'contact_email', 'customer_email'];
            var email = null;

            for (var i = 0; i < emailFields.length; i++) {
                var val = formData.get(emailFields[i]);
                if (val && val.indexOf('@') !== -1) {
                    email = val.toLowerCase().trim();
                    break;
                }
            }

            // Also check all fields for email pattern
            if (!email) {
                formData.forEach(function(value) {
                    if (!email && value && typeof value === 'string' && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) {
                        email = value.toLowerCase().trim();
                    }
                });
            }

            if (!email) return { matched: false };

            var emailDomain = email.split('@')[1];

            for (var j = 0; j < blacklist.length; j++) {
                var pattern = blacklist[j];
                if (pattern.startsWith('*@')) {
                    // Wildcard domain match
                    if (emailDomain === pattern.substring(2)) {
                        return { matched: true, email: email };
                    }
                } else if (pattern === email || pattern === emailDomain) {
                    return { matched: true, email: email };
                }
            }

            return { matched: false };
        },

        /**
         * Domain Blacklist Check
         */
        checkDomainBlacklist: function(content, formData) {
            var blacklistStr = typeof this.config.domainBlacklist === 'string' ? this.config.domainBlacklist : '';
            var blacklist = blacklistStr.toLowerCase().split(',').map(function(d) {
                return d.trim();
            }).filter(Boolean);

            if (blacklist.length === 0) return { matched: false };

            // Extract domains from URLs in content
            var urlPattern = /https?:\/\/([^\/\s<>"']+)/gi;
            var match;
            var domains = [];

            while ((match = urlPattern.exec(content)) !== null) {
                domains.push(match[1].toLowerCase());
            }

            // Also extract email domain
            formData.forEach(function(value) {
                if (value && typeof value === 'string' && value.indexOf('@') !== -1) {
                    var emailMatch = value.match(/@([^\s@]+)/);
                    if (emailMatch) {
                        domains.push(emailMatch[1].toLowerCase());
                    }
                }
            });

            for (var i = 0; i < domains.length; i++) {
                for (var j = 0; j < blacklist.length; j++) {
                    if (domains[i] === blacklist[j] || domains[i].endsWith('.' + blacklist[j])) {
                        return { matched: true, domain: domains[i] };
                    }
                }
            }

            return { matched: false };
        },

        /**
         * Gibberish Detection
         */
        checkGibberish: function(formData) {
            var result = {
                score: 0,
                reasons: [],
                details: {}
            };

            var checkFields = ['name', 'first', 'last', 'your-name', 'subject', 'your-subject', 'message', 'comment', 'company', 'city', 'address'];

            var self = this;
            formData.forEach(function(value, key) {
                if (result.score > 0) return; // Already found gibberish

                // Skip internal fields
                if (key.startsWith('_oh_') || key.startsWith('_wp')) return;

                // Skip email and phone fields
                if (/email|phone|tel|mail|fax/i.test(key)) return;

                // Check if this field should be examined
                var shouldCheck = checkFields.some(function(field) {
                    return key.toLowerCase().indexOf(field) !== -1;
                });

                if (!shouldCheck) return;

                var text = (typeof value === 'string') ? value.trim() : '';
                if (text.length < 6) return;

                // Skip if looks like valid data
                if (/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(text)) return; // Email
                if (/^https?:\/\//.test(text)) return; // URL
                if (/^[\d\s\+\-\(\)]+$/.test(text)) return; // Phone number

                if (self.isObviousGibberish(text)) {
                    result.score = 85;
                    result.reasons.push('Gibberish detected in: ' + key);
                    result.details.gibberishField = key;
                    result.details.gibberishValue = text.substring(0, 50);
                }
            });

            return result;
        },

        /**
         * Check if text is obvious gibberish
         */
        isObviousGibberish: function(text) {
            var condensed = text.replace(/\s+/g, '');
            var len = condensed.length;

            if (len < 6) return false;

            // Check 1: Very low character variety (keyboard mashing like "safasafafafaffff")
            var chars = condensed.toLowerCase().split('');
            var uniqueChars = chars.filter(function(c, i, arr) {
                return arr.indexOf(c) === i;
            }).length;
            var varietyRatio = uniqueChars / len;

            if (len >= 8 && varietyRatio < 0.35) {
                return true;
            }

            // Check 2: Repeated short patterns (like "asdfasdf", "qwer qwer")
            var lower = condensed.toLowerCase();
            if (len >= 8) {
                for (var patternLen = 2; patternLen <= 4; patternLen++) {
                    var pattern = lower.substring(0, patternLen);
                    var repeated = '';
                    for (var k = 0; k < Math.ceil(len / patternLen); k++) {
                        repeated += pattern;
                    }
                    if (repeated.substring(0, len) === lower) {
                        return true;
                    }
                }
            }

            // Check 3: Very low vowel ratio
            var vowelCount = (condensed.match(/[aeiouAEIOU]/g) || []).length;
            var vowelRatio = vowelCount / len;
            if (vowelRatio < 0.08 && len > 6) {
                return true;
            }

            // Check 4: Long consonant sequences (5+ consonants in a row)
            if (/[bcdfghjklmnpqrstvwxzBCDFGHJKLMNPQRSTVWXZ]{5,}/.test(condensed)) {
                return true;
            }

            // Check 5: Random mixed case pattern (like "QGNItWfBJmqYrUNcxjYOmdZ")
            var uppercase = (condensed.match(/[A-Z]/g) || []).length;
            var lowercase = (condensed.match(/[a-z]/g) || []).length;
            if (uppercase > 3 && lowercase > 3 && len > 10) {
                var midUpper = (condensed.match(/(?<=[a-z])[A-Z]/g) || []).length;
                if (midUpper > 3) {
                    return true;
                }
            }

            return false;
        },

        /**
         * Heuristic analysis
         */
        runHeuristics: function(content) {
            var result = {
                score: 0,
                reasons: [],
                details: {}
            };

            // Link density
            var links = (content.match(/https?:\/\/[^\s]+/gi) || []);
            if (links.length > this.config.maxLinks) {
                result.score += 40;
                result.reasons.push('High link density (' + links.length + ' links)');
                result.details.linkCount = links.length;
            }

            // Spam patterns
            var patterns = [
                { regex: /\b(viagra|cialis|casino|lottery|winner|congratulations|you\s+won)\b/i, points: 30, name: 'Spam words' },
                { regex: /\b(crypto|bitcoin|investment\s+opportunity|make\s+money\s+fast)\b/i, points: 25, name: 'Scam terms' },
                { regex: /\b(click\s+here|act\s+now|limited\s+time|free\s+offer)\b/i, points: 20, name: 'Marketing spam' },
                { regex: /\b(SEO\s+service|backlink|rank\s+#1|first\s+page\s+google)\b/i, points: 35, name: 'SEO spam' },
                { regex: /\b(dear\s+sir|dear\s+madam|dear\s+friend|to\s+whom\s+it\s+may\s+concern)\b/i, points: 15, name: 'Generic greeting' },
                { regex: /(.)\1{5,}/, points: 15, name: 'Character repetition' },
                { regex: /[A-Z]{15,}/, points: 10, name: 'Excessive caps' }
            ];

            patterns.forEach(function(p) {
                if (p.regex.test(content)) {
                    result.score += p.points;
                    result.reasons.push(p.name + ' detected');
                    result.details.patternMatch = true;
                }
            });

            // Injection attempts
            if (/<[^>]*script|<[^>]*iframe|<[^>]*object|javascript:/i.test(content)) {
                result.score += 50;
                result.reasons.push('Potential code injection');
                result.details.injection = true;
            }

            // Multiple emails
            var emails = (content.match(/[\w.-]+@[\w.-]+\.\w+/g) || []);
            if (emails.length > 2) {
                result.score += 25;
                result.reasons.push('Multiple email addresses (' + emails.length + ')');
                result.details.emailCount = emails.length;
            }

            return result;
        },

        /**
         * Handle spam detection
         */
        handleSpam: function(form, result) {
            // Track blocked entries
            if (result.details.emailBlacklisted) {
                this.addBlockedEntry('email', result.details.emailBlacklisted, 'Email blacklisted');
            }
            if (result.details.domainBlacklisted) {
                this.addBlockedEntry('domain', result.details.domainBlacklisted, 'Domain blacklisted');
            }
            if (result.details.blacklistMatch) {
                this.addBlockedEntry('keyword', result.details.blacklistMatch, 'Keyword matched');
            }
            if (result.details.gibberishField) {
                this.addBlockedEntry('gibberish', result.details.gibberishValue, 'Gibberish in ' + result.details.gibberishField);
            }
            if (result.details.honeypot) {
                this.addBlockedEntry('honeypot', 'form_submission', 'Honeypot triggered');
            }

            this.incrementSpamCount();

            if (this.config.showErrorMessage && form) {
                this.showError(form, this.config.errorMessage);
            }

            this.checkEscalation();

            this.log('Spam blocked:', result);
        },

        /**
         * Inject honeypot field
         */
        injectHoneypot: function(form) {
            if (form.querySelector('[data-oh-honeypot]')) return;

            this.createHoneypotField(form, this.config.honeypotFieldName);
        },

        /**
         * Inject multiple honeypot fields
         */
        injectMultipleHoneypots: function(form) {
            var self = this;
            var additionalNames = this.config.additionalHoneypotNames || [];

            additionalNames.forEach(function(name) {
                if (!form.querySelector('[name="' + name + '"]')) {
                    self.createHoneypotField(form, name);
                }
            });
        },

        /**
         * Create a honeypot field
         */
        createHoneypotField: function(form, fieldName) {
            var wrapper = document.createElement('div');
            wrapper.style.cssText = 'position:absolute;left:-9999px;top:-9999px;opacity:0;pointer-events:none;height:0;overflow:hidden;';
            wrapper.setAttribute('aria-hidden', 'true');

            var input = document.createElement('input');
            input.type = 'text';
            input.name = fieldName;
            input.tabIndex = -1;
            input.autocomplete = 'nope';
            input.setAttribute('data-oh-honeypot', 'true');

            wrapper.appendChild(input);
            form.appendChild(wrapper);
        },

        /**
         * Inject timestamp field
         */
        injectTimestamp: function(form) {
            if (form.querySelector('[name="_oh_ts"]')) return;

            var input = document.createElement('input');
            input.type = 'hidden';
            input.name = '_oh_ts';
            input.value = Math.floor(Date.now() / 1000);

            form.appendChild(input);
        },

        /**
         * Inject CSRF token
         */
        injectCSRFToken: function(form) {
            if (form.querySelector('[name="_oh_tk"]')) return;

            var input = document.createElement('input');
            input.type = 'hidden';
            input.name = '_oh_tk';
            input.value = this.csrfToken;

            form.appendChild(input);
        },

        /**
         * Set verification cookie
         */
        setVerifyCookie: function() {
            var d = new Date();
            d.setTime(d.getTime() + (24 * 60 * 60 * 1000)); // 24 hours
            var value = this.csrfToken.substring(0, 8) + '-' + Math.random().toString(36).substring(2, 10);
            document.cookie = '_oh_verify=' + value + ';expires=' + d.toUTCString() + ';path=/;SameSite=Lax';
            this.verifyCookie = value;
        },

        /**
         * Get cookie value
         */
        getCookie: function(name) {
            var match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
            return match ? match[2] : null;
        },

        /**
         * Show error message
         */
        showError: function(form, message) {
            var existing = form.querySelector('.oh-error');
            if (existing) existing.remove();

            var div = document.createElement('div');
            div.className = 'oh-error';
            div.style.cssText = 'background:#f8d7da;color:#721c24;padding:12px 16px;border-radius:4px;margin-bottom:16px;font-size:14px;border:1px solid #f5c6cb;';
            div.textContent = message;

            form.insertBefore(div, form.firstChild);

            setTimeout(function() {
                if (div.parentNode) div.remove();
            }, 5000);
        },

        /**
         * Extract content from form data
         */
        extractContent: function(formData) {
            var content = '';
            var exclude = ['_oh_ts', '_oh_tk', this.config.honeypotFieldName];

            // Add additional honeypot names to exclude
            if (this.config.additionalHoneypotNames) {
                exclude = exclude.concat(this.config.additionalHoneypotNames);
            }

            formData.forEach(function(value, key) {
                if (exclude.indexOf(key) === -1 && !key.startsWith('_oh_')) {
                    content += ' ' + value;
                }
            });

            return content.trim();
        },

        /**
         * Check if form should be excluded
         */
        shouldExcludeForm: function(form) {
            var self = this;

            return this.config.excludeSelectors.some(function(selector) {
                return form.matches(selector) || form.closest(selector);
            });
        },

        /**
         * Observe DOM for dynamically added forms
         */
        observeDOM: function() {
            var self = this;

            if (!window.MutationObserver) return;

            var observer = new MutationObserver(function(mutations) {
                mutations.forEach(function(mutation) {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1) {
                            if (node.tagName === 'FORM' && !self.shouldExcludeForm(node)) {
                                self.protect(node);
                            }

                            var forms = node.querySelectorAll ? node.querySelectorAll('form') : [];
                            forms.forEach(function(form) {
                                if (!self.shouldExcludeForm(form)) {
                                    self.protect(form);
                                }
                            });
                        }
                    });
                });
            });

            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
        },

        /**
         * Rate limiting - check if rate limited
         */
        isRateLimited: function() {
            var now = Date.now();
            var windowMs = this.config.rateLimitWindow * 1000;

            this.submissionTimestamps = this.submissionTimestamps.filter(function(ts) {
                return (now - ts) < windowMs;
            });

            return this.submissionTimestamps.length >= this.config.rateLimitSubmissions;
        },

        /**
         * Track submission for rate limiting
         */
        trackSubmission: function() {
            this.submissionTimestamps.push(Date.now());
            this.saveSubmissionTimestamps();
        },

        /**
         * Load submission timestamps from storage
         */
        loadSubmissionTimestamps: function() {
            try {
                var stored = sessionStorage.getItem('oh_submissions');
                if (stored) {
                    var now = Date.now();
                    var windowMs = this.config.rateLimitWindow * 1000;

                    this.submissionTimestamps = JSON.parse(stored).filter(function(ts) {
                        return (now - ts) < windowMs;
                    });
                }
            } catch (e) {}
        },

        /**
         * Save submission timestamps to storage
         */
        saveSubmissionTimestamps: function() {
            try {
                sessionStorage.setItem('oh_submissions', JSON.stringify(this.submissionTimestamps));
            } catch (e) {}
        },

        /**
         * Spam count management
         */
        incrementSpamCount: function() {
            var now = Date.now();
            this.spamTimestamps.push(now);

            var windowMs = this.config.escalationWindow * 60 * 1000;
            this.spamTimestamps = this.spamTimestamps.filter(function(ts) {
                return (now - ts) < windowMs;
            });

            this.saveSpamCount();
        },

        loadSpamCount: function() {
            try {
                var stored = sessionStorage.getItem('oh_spam');
                if (stored) {
                    var now = Date.now();
                    var windowMs = this.config.escalationWindow * 60 * 1000;

                    this.spamTimestamps = JSON.parse(stored).filter(function(ts) {
                        return (now - ts) < windowMs;
                    });
                }
            } catch (e) {}
        },

        saveSpamCount: function() {
            try {
                sessionStorage.setItem('oh_spam', JSON.stringify(this.spamTimestamps));
            } catch (e) {}
        },

        checkEscalation: function() {
            if (!this.config.escalationEnabled) return;

            if (this.spamTimestamps.length >= this.config.escalationThreshold) {
                this.showEscalationBanner();
            }
        },

        showEscalationBanner: function() {
            if (document.querySelector('.oh-escalation')) return;

            var banner = document.createElement('div');
            banner.className = 'oh-escalation';
            banner.style.cssText = 'position:fixed;bottom:20px;right:20px;background:linear-gradient(135deg,#2c5aa0,#1e3d6b);color:#fff;padding:16px 20px;border-radius:8px;box-shadow:0 4px 20px rgba(0,0,0,.2);z-index:99999;max-width:300px;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;font-size:14px;';

            banner.innerHTML = '<div style="display:flex;gap:12px;align-items:flex-start;">' +
                '<span style="font-size:20px;">&#128737;</span>' +
                '<div style="flex:1;">' +
                '<strong style="display:block;margin-bottom:4px;">Spam Issues?</strong>' +
                '<p style="margin:0 0 10px;font-size:13px;opacity:.9;">Need advanced protection?</p>' +
                '<a href="' + this.config.supportUrl + '" target="_blank" style="display:inline-block;background:#fff;color:#2c5aa0;padding:6px 14px;border-radius:4px;text-decoration:none;font-weight:600;font-size:12px;">Get Help</a>' +
                '</div>' +
                '<button onclick="this.closest(\'.oh-escalation\').remove()" style="background:none;border:none;color:#fff;cursor:pointer;font-size:18px;line-height:1;padding:0;opacity:.7;">&times;</button>' +
                '</div>';

            document.body.appendChild(banner);
        },

        /**
         * Utilities
         */
        generateToken: function() {
            var arr = new Uint8Array(16);
            if (window.crypto && window.crypto.getRandomValues) {
                window.crypto.getRandomValues(arr);
            } else {
                for (var i = 0; i < 16; i++) {
                    arr[i] = Math.floor(Math.random() * 256);
                }
            }
            return Array.from(arr, function(b) { return b.toString(16).padStart(2, '0'); }).join('');
        },

        simpleHash: function(str) {
            var hash = 0;
            for (var i = 0; i < str.length; i++) {
                hash = ((hash << 5) - hash) + str.charCodeAt(i);
                hash |= 0;
            }
            return Math.abs(hash);
        },

        log: function() {
            if (this.config.debug) {
                console.log.apply(console, ['[Open Honeypot]'].concat(Array.prototype.slice.call(arguments)));
            }
        },

        // =========================================================================
        // BLOCKED ENTRIES MANAGEMENT (localStorage-based)
        // Since client-side JS cannot access visitor IPs, we track blocked emails,
        // keywords, gibberish patterns, and other spam indicators.
        // =========================================================================

        /**
         * Add an entry to the blocked list
         */
        addBlockedEntry: function(type, value, reason) {
            var entries = this.getBlockedEntries();
            var entry = {
                type: type, // 'email', 'keyword', 'gibberish', 'honeypot', 'domain'
                value: value,
                reason: reason,
                timestamp: Date.now(),
                blocked: true
            };

            // Check if already exists
            var exists = entries.some(function(e) {
                return e.type === type && e.value === value;
            });

            if (!exists) {
                entries.push(entry);
                this.saveBlockedEntries(entries);
                this.log('Blocked entry added:', entry);
            }
        },

        /**
         * Get all blocked entries
         */
        getBlockedEntries: function() {
            try {
                var stored = localStorage.getItem('oh_blocked_entries');
                return stored ? JSON.parse(stored) : [];
            } catch (e) {
                return [];
            }
        },

        /**
         * Save blocked entries
         */
        saveBlockedEntries: function(entries) {
            try {
                localStorage.setItem('oh_blocked_entries', JSON.stringify(entries));
            } catch (e) {
                this.log('Failed to save blocked entries:', e);
            }
        },

        /**
         * View all blocked entries (console-friendly)
         * Usage: OpenHoneypot.viewBlockedEntries()
         */
        viewBlockedEntries: function() {
            var entries = this.getBlockedEntries();
            if (entries.length === 0) {
                console.log('[Open Honeypot] No blocked entries found.');
                return [];
            }

            console.log('[Open Honeypot] Blocked Entries (' + entries.length + '):');
            console.table(entries.map(function(e, i) {
                return {
                    index: i,
                    type: e.type,
                    value: e.value,
                    reason: e.reason,
                    date: new Date(e.timestamp).toLocaleString(),
                    blocked: e.blocked
                };
            }));

            return entries;
        },

        /**
         * Unblock an entry by index (keeps entry but marks as unblocked)
         * Usage: OpenHoneypot.unblockEntry(0)
         */
        unblockEntry: function(index) {
            var entries = this.getBlockedEntries();
            if (index < 0 || index >= entries.length) {
                console.log('[Open Honeypot] Invalid index. Use viewBlockedEntries() to see all entries.');
                return false;
            }

            var entry = entries[index];
            entry.blocked = false;
            this.saveBlockedEntries(entries);
            console.log('[Open Honeypot] Unblocked entry:', entry.type, '-', entry.value);
            return true;
        },

        /**
         * Remove an entry completely
         * Usage: OpenHoneypot.removeEntry(0)
         */
        removeEntry: function(index) {
            var entries = this.getBlockedEntries();
            if (index < 0 || index >= entries.length) {
                console.log('[Open Honeypot] Invalid index. Use viewBlockedEntries() to see all entries.');
                return false;
            }

            var removed = entries.splice(index, 1)[0];
            this.saveBlockedEntries(entries);
            console.log('[Open Honeypot] Removed entry:', removed.type, '-', removed.value);
            return true;
        },

        /**
         * Clear all blocked entries
         * Usage: OpenHoneypot.clearBlockedEntries()
         */
        clearBlockedEntries: function() {
            try {
                localStorage.removeItem('oh_blocked_entries');
                console.log('[Open Honeypot] All blocked entries cleared.');
                return true;
            } catch (e) {
                return false;
            }
        },

        /**
         * Check if a value is blocked
         */
        isBlocked: function(type, value) {
            var entries = this.getBlockedEntries();
            return entries.some(function(e) {
                return e.type === type && e.value === value && e.blocked;
            });
        },

        /**
         * Export blocked entries as JSON (for backup/transfer)
         * Usage: OpenHoneypot.exportBlockedEntries()
         */
        exportBlockedEntries: function() {
            var entries = this.getBlockedEntries();
            var json = JSON.stringify(entries, null, 2);
            console.log('[Open Honeypot] Blocked Entries JSON:');
            console.log(json);
            return json;
        },

        /**
         * Import blocked entries from JSON
         * Usage: OpenHoneypot.importBlockedEntries('[...]')
         */
        importBlockedEntries: function(json) {
            try {
                var entries = JSON.parse(json);
                if (!Array.isArray(entries)) {
                    throw new Error('Invalid format');
                }
                this.saveBlockedEntries(entries);
                console.log('[Open Honeypot] Imported ' + entries.length + ' entries.');
                return true;
            } catch (e) {
                console.log('[Open Honeypot] Failed to import:', e.message);
                return false;
            }
        }
    };

    // Auto-initialize if not in module environment
    if (typeof module === 'undefined' && typeof define === 'undefined') {
        OpenHoneypot.init();
    }

    return OpenHoneypot;
}));
