/**
 * Open Honeypot Anti-Spam Suite for Shopify
 * Version: 2.0.1
 *
 * Lightweight honeypot and anti-spam protection for Shopify contact forms.
 * Add this script to your theme.liquid or via the Script Editor app.
 *
 * Installation:
 * 1. In Shopify Admin, go to Online Store > Themes > Edit Code
 * 2. Under Assets, click "Add a new asset" and upload this file
 * 3. In theme.liquid, add before </body>: {{ 'open-honeypot-shopify.js' | asset_url | script_tag }}
 * 4. Configure options below if needed
 *
 * Learn more: https://github.com/user/open-honeypot
 * Professional support: https://www.mantasdigital.com/
 */

(function() {
    'use strict';

    // =========================================================================
    // CONFIGURATION - Customize these settings
    // =========================================================================

    var OpenHoneypotConfig = {
        // Honeypot Settings
        honeypotEnabled: true,
        honeypotFieldName: 'company_extension', // Realistic field name to trick bots
        multipleHoneypots: false, // Add multiple honeypot fields
        additionalHoneypotNames: ['middle_name', 'fax_number'], // Extra honeypot field names

        // Timestamp Settings
        timestampEnabled: true,
        minSubmissionTime: 2, // Minimum seconds before form can be submitted
        maxSubmissionTime: 3600, // Maximum seconds (1 hour) - form expires after this

        // Cookie Verification (proves JavaScript executed)
        cookieCheckEnabled: false, // Set to true to enable cookie verification

        // Referer Check
        requireReferer: false, // Set to true to require document.referrer

        // Rate Limiting
        rateLimitEnabled: false,
        rateLimitSubmissions: 5, // Max submissions allowed
        rateLimitWindow: 60, // Time window in seconds

        // Character Filtering
        blockCyrillic: false, // Block Russian/Cyrillic characters
        blockHanzi: false, // Block Chinese characters
        blockBengali: false, // Block South Asian scripts (Bengali, Devanagari, Tamil, etc.)
        blockLatin: false, // Block Latin characters (use with caution!)

        // Keyword Blacklist (comma-separated)
        keywordBlacklist: 'viagra,cialis,casino,lottery,crypto investment,seo service',

        // Email Blacklist (comma-separated emails or patterns like *@spam.com)
        emailBlacklist: '',

        // Domain Blacklist (comma-separated domains found in URLs/content)
        domainBlacklist: '',

        // Gibberish Detection (keyboard mashing, random text)
        gibberishDetectionEnabled: true,

        // Heuristic Analysis
        heuristicEnabled: true,
        maxLinks: 3, // Maximum allowed links in submission

        // Spam Threshold (0-100)
        spamThreshold: 70,

        // Form Selectors (customize based on your theme)
        formSelectors: [
            'form[action*="/contact"]',
            'form.contact-form',
            '#contact_form',
            '.shopify-challenge__container form',
            'form[action*="contact#contact_form"]'
        ],

        // Error Messages (customize for your language)
        errorMessage: 'Your submission could not be processed. Please try again or contact us directly.',

        // Logging (set to true during testing)
        debug: false,

        // Escalation Settings
        escalationEnabled: true,
        escalationThreshold: 3, // Spam count before showing support link
        escalationWindow: 60, // Minutes
        supportUrl: 'https://www.mantasdigital.com/'
    };

    // =========================================================================
    // REALISTIC HONEYPOT FIELD NAMES
    // =========================================================================

    var honeypotNames = [
        'middle_name',
        'user_verify_id',
        'extra_billing_note',
        'secondary_phone',
        'alt_email_address',
        'company_extension',
        'fax_number',
        'preferred_contact_time',
        'referral_source_detail',
        'additional_notes_field'
    ];

    // =========================================================================
    // CORE DETECTION ENGINE
    // =========================================================================

    var OpenHoneypot = {
        spamCount: 0,
        spamTimestamps: [],
        submissionTimestamps: [],
        csrfToken: null,
        verifyCookie: null,

        init: function() {
            var self = this;

            // Generate CSRF token
            this.csrfToken = this.generateToken();

            // Generate and set verification cookie if enabled
            if (OpenHoneypotConfig.cookieCheckEnabled) {
                this.setVerifyCookie();
            }

            // Wait for DOM ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', function() {
                    self.setup();
                });
            } else {
                this.setup();
            }

            this.log('Open Honeypot initialized');
        },

        setup: function() {
            var self = this;

            // Load spam count from session storage
            this.loadSpamCount();
            this.loadSubmissionTimestamps();

            // Find and protect all matching forms
            OpenHoneypotConfig.formSelectors.forEach(function(selector) {
                var forms = document.querySelectorAll(selector);
                forms.forEach(function(form) {
                    self.protectForm(form);
                });
            });

            // Check for escalation condition
            this.checkEscalation();
        },

        protectForm: function(form) {
            var self = this;

            // Skip if already protected
            if (form.hasAttribute('data-oh-protected')) {
                return;
            }

            form.setAttribute('data-oh-protected', 'true');

            // Inject honeypot field(s)
            if (OpenHoneypotConfig.honeypotEnabled) {
                this.injectHoneypot(form);

                // Add multiple honeypots if enabled
                if (OpenHoneypotConfig.multipleHoneypots) {
                    this.injectMultipleHoneypots(form);
                }
            }

            // Inject timestamp field
            if (OpenHoneypotConfig.timestampEnabled) {
                this.injectTimestamp(form);
            }

            // Inject CSRF token
            this.injectCSRFToken(form);

            // Intercept form submission
            form.addEventListener('submit', function(e) {
                var result = self.validateSubmission(form);

                if (result.isSpam) {
                    e.preventDefault();
                    e.stopPropagation();

                    self.handleSpam(form, result);
                    return false;
                }

                // Track submission for rate limiting
                if (OpenHoneypotConfig.rateLimitEnabled) {
                    self.trackSubmission();
                }

                self.log('Form submission allowed', result);
            });

            this.log('Form protected:', form);
        },

        injectHoneypot: function(form) {
            var fieldName = OpenHoneypotConfig.honeypotFieldName;

            // Check if honeypot already exists
            if (form.querySelector('[name="' + fieldName + '"]')) {
                return;
            }

            this.createHoneypotField(form, fieldName);
        },

        injectMultipleHoneypots: function(form) {
            var self = this;
            var additionalNames = OpenHoneypotConfig.additionalHoneypotNames || [];

            additionalNames.forEach(function(name) {
                if (!form.querySelector('[name="' + name + '"]')) {
                    self.createHoneypotField(form, name);
                }
            });
        },

        createHoneypotField: function(form, fieldName) {
            var wrapper = document.createElement('div');
            wrapper.style.cssText = 'position:absolute;left:-9999px;top:-9999px;opacity:0;pointer-events:none;height:0;overflow:hidden;';
            wrapper.setAttribute('aria-hidden', 'true');

            var label = document.createElement('label');
            label.setAttribute('for', 'oh_' + fieldName);
            label.textContent = 'Leave this field empty';

            var input = document.createElement('input');
            input.type = 'text';
            input.name = fieldName;
            input.id = 'oh_' + fieldName;
            input.tabIndex = -1;
            input.autocomplete = 'nope';
            input.setAttribute('data-oh-honeypot', 'true');

            wrapper.appendChild(label);
            wrapper.appendChild(input);
            form.appendChild(wrapper);
        },

        injectTimestamp: function(form) {
            if (form.querySelector('[name="_oh_timestamp"]')) {
                return;
            }

            var input = document.createElement('input');
            input.type = 'hidden';
            input.name = '_oh_timestamp';
            input.value = Math.floor(Date.now() / 1000);

            form.appendChild(input);
        },

        injectCSRFToken: function(form) {
            if (form.querySelector('[name="_oh_token"]')) {
                return;
            }

            var input = document.createElement('input');
            input.type = 'hidden';
            input.name = '_oh_token';
            input.value = this.csrfToken;

            form.appendChild(input);
        },

        setVerifyCookie: function() {
            var d = new Date();
            d.setTime(d.getTime() + (24 * 60 * 60 * 1000)); // 24 hours
            var value = this.csrfToken.substring(0, 8) + '-' + Math.random().toString(36).substring(2, 10);
            document.cookie = '_oh_verify=' + value + ';expires=' + d.toUTCString() + ';path=/;SameSite=Lax';
            this.verifyCookie = value;
        },

        getCookie: function(name) {
            var match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
            return match ? match[2] : null;
        },

        validateSubmission: function(form) {
            var result = {
                isSpam: false,
                score: 0,
                reasons: [],
                details: {}
            };

            var formData = new FormData(form);
            var content = this.getFormContent(formData);

            // 1. Rate Limit Check
            if (OpenHoneypotConfig.rateLimitEnabled && this.isRateLimited()) {
                result.score += 100;
                result.reasons.push('Rate limit exceeded');
                result.details.rateLimited = true;
            }

            // 2. Referer Check
            if (OpenHoneypotConfig.requireReferer && !document.referrer) {
                result.score += 30;
                result.reasons.push('Missing referer');
                result.details.noReferer = true;
            }

            // 3. Cookie Verification Check
            if (OpenHoneypotConfig.cookieCheckEnabled) {
                var cookie = this.getCookie('_oh_verify');
                if (!cookie) {
                    result.score += 40;
                    result.reasons.push('Missing verification cookie');
                    result.details.noCookie = true;
                }
            }

            // 4. Honeypot Check
            if (OpenHoneypotConfig.honeypotEnabled) {
                var honeypotValue = formData.get(OpenHoneypotConfig.honeypotFieldName);
                if (honeypotValue && honeypotValue.trim() !== '') {
                    result.score += 100;
                    result.reasons.push('Honeypot field filled');
                    result.details.honeypot = true;
                }

                // Check multiple honeypots
                if (OpenHoneypotConfig.multipleHoneypots) {
                    var additionalNames = OpenHoneypotConfig.additionalHoneypotNames || [];
                    for (var i = 0; i < additionalNames.length; i++) {
                        var val = formData.get(additionalNames[i]);
                        if (val && val.trim() !== '') {
                            result.score += 100;
                            result.reasons.push('Additional honeypot filled: ' + additionalNames[i]);
                            result.details.multipleHoneypot = true;
                            break;
                        }
                    }
                }
            }

            // 5. Timestamp Check
            if (OpenHoneypotConfig.timestampEnabled) {
                var timestamp = parseInt(formData.get('_oh_timestamp'), 10);
                var currentTime = Math.floor(Date.now() / 1000);
                var elapsedTime = currentTime - timestamp;

                if (elapsedTime < OpenHoneypotConfig.minSubmissionTime) {
                    result.score += 80;
                    result.reasons.push('Submitted too fast (' + elapsedTime + 's)');
                    result.details.tooFast = true;
                }

                if (elapsedTime > OpenHoneypotConfig.maxSubmissionTime) {
                    result.score += 40;
                    result.reasons.push('Form session expired');
                    result.details.expired = true;
                }
            }

            // 6. CSRF Token Check
            var submittedToken = formData.get('_oh_token');
            if (submittedToken !== this.csrfToken) {
                result.score += 50;
                result.reasons.push('Invalid security token');
                result.details.invalidToken = true;
            }

            // 7. Email Blacklist Check
            if (OpenHoneypotConfig.emailBlacklist) {
                var emailResult = this.checkEmailBlacklist(formData);
                if (emailResult.matched) {
                    result.score += 100;
                    result.reasons.push('Email blacklisted: ' + emailResult.email);
                    result.details.emailBlacklisted = emailResult.email;
                }
            }

            // 8. Domain Blacklist Check
            if (OpenHoneypotConfig.domainBlacklist) {
                var domainResult = this.checkDomainBlacklist(content, formData);
                if (domainResult.matched) {
                    result.score += 90;
                    result.reasons.push('Domain blacklisted: ' + domainResult.domain);
                    result.details.domainBlacklisted = domainResult.domain;
                }
            }

            // 9. Character Filtering
            // Cyrillic
            if (OpenHoneypotConfig.blockCyrillic) {
                if (/[\u0400-\u04FF]/.test(content)) {
                    result.score += 60;
                    result.reasons.push('Contains Cyrillic characters');
                    result.details.cyrillic = true;
                }
            }

            // Chinese
            if (OpenHoneypotConfig.blockHanzi) {
                if (/[\u4E00-\u9FFF]/.test(content)) {
                    result.score += 60;
                    result.reasons.push('Contains Chinese characters');
                    result.details.hanzi = true;
                }
            }

            // South Asian Scripts (Bengali, Devanagari, Tamil, Telugu, etc.)
            if (OpenHoneypotConfig.blockBengali) {
                if (/[\u0980-\u09FF\u0900-\u097F\u0B80-\u0BFF\u0C00-\u0C7F\u0C80-\u0CFF\u0D00-\u0D7F\u0A80-\u0AFF\u0B00-\u0B7F\u0A00-\u0A7F]/.test(content)) {
                    result.score += 60;
                    result.reasons.push('Contains South Asian script characters');
                    result.details.bengali = true;
                }
            }

            // Latin (use with caution)
            if (OpenHoneypotConfig.blockLatin) {
                if (/[a-zA-Z]/.test(content)) {
                    result.score += 60;
                    result.reasons.push('Contains Latin characters');
                    result.details.latin = true;
                }
            }

            // 10. Keyword Blacklist
            if (OpenHoneypotConfig.keywordBlacklist) {
                var blacklist = OpenHoneypotConfig.keywordBlacklist.toLowerCase().split(',').map(function(k) {
                    return k.trim();
                });
                var contentLower = content.toLowerCase();

                for (var j = 0; j < blacklist.length; j++) {
                    if (blacklist[j] && contentLower.indexOf(blacklist[j]) !== -1) {
                        result.score += 90;
                        result.reasons.push('Blacklisted keyword: ' + blacklist[j]);
                        result.details.blacklistMatch = blacklist[j];
                        break;
                    }
                }
            }

            // 11. Gibberish Detection
            if (OpenHoneypotConfig.gibberishDetectionEnabled) {
                var gibberishResult = this.checkGibberish(formData);
                if (gibberishResult.score > 0) {
                    result.score += gibberishResult.score;
                    result.reasons = result.reasons.concat(gibberishResult.reasons);
                    Object.assign(result.details, gibberishResult.details);
                }
            }

            // 12. Heuristic Analysis
            if (OpenHoneypotConfig.heuristicEnabled) {
                var heuristicResult = this.heuristicAnalysis(content);
                result.score += heuristicResult.score;
                result.reasons = result.reasons.concat(heuristicResult.reasons);
                Object.assign(result.details, heuristicResult.details);
            }

            // Determine if spam
            result.isSpam = result.score >= OpenHoneypotConfig.spamThreshold;

            this.log('Validation result:', result);

            return result;
        },

        // Email Blacklist Check
        checkEmailBlacklist: function(formData) {
            var blacklist = OpenHoneypotConfig.emailBlacklist.toLowerCase().split(',').map(function(e) {
                return e.trim();
            }).filter(Boolean);

            if (blacklist.length === 0) return { matched: false };

            // Find email fields
            var emailFields = ['email', 'your-email', 'mail', 'contact_email', 'customer_email'];
            var email = null;

            for (var i = 0; i < emailFields.length; i++) {
                var val = formData.get(emailFields[i]);
                if (val && val.indexOf('@') !== -1) {
                    email = val.toLowerCase().trim();
                    break;
                }
            }

            // Also check all fields for email pattern
            if (!email) {
                formData.forEach(function(value, key) {
                    if (!email && value && typeof value === 'string' && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) {
                        email = value.toLowerCase().trim();
                    }
                });
            }

            if (!email) return { matched: false };

            var emailDomain = email.split('@')[1];

            for (var j = 0; j < blacklist.length; j++) {
                var pattern = blacklist[j];
                if (pattern.startsWith('*@')) {
                    // Wildcard domain match
                    if (emailDomain === pattern.substring(2)) {
                        return { matched: true, email: email };
                    }
                } else if (pattern === email || pattern === emailDomain) {
                    return { matched: true, email: email };
                }
            }

            return { matched: false };
        },

        // Domain Blacklist Check
        checkDomainBlacklist: function(content, formData) {
            var blacklist = OpenHoneypotConfig.domainBlacklist.toLowerCase().split(',').map(function(d) {
                return d.trim();
            }).filter(Boolean);

            if (blacklist.length === 0) return { matched: false };

            // Extract domains from URLs in content
            var urlPattern = /https?:\/\/([^\/\s<>"']+)/gi;
            var match;
            var domains = [];

            while ((match = urlPattern.exec(content)) !== null) {
                domains.push(match[1].toLowerCase());
            }

            // Also extract email domain
            formData.forEach(function(value) {
                if (value && typeof value === 'string' && value.indexOf('@') !== -1) {
                    var emailMatch = value.match(/@([^\s@]+)/);
                    if (emailMatch) {
                        domains.push(emailMatch[1].toLowerCase());
                    }
                }
            });

            for (var i = 0; i < domains.length; i++) {
                for (var j = 0; j < blacklist.length; j++) {
                    if (domains[i] === blacklist[j] || domains[i].endsWith('.' + blacklist[j])) {
                        return { matched: true, domain: domains[i] };
                    }
                }
            }

            return { matched: false };
        },

        // Gibberish Detection
        checkGibberish: function(formData) {
            var result = {
                score: 0,
                reasons: [],
                details: {}
            };

            var checkFields = ['name', 'first', 'last', 'your-name', 'subject', 'your-subject', 'message', 'comment', 'company', 'city', 'address'];

            var self = this;
            formData.forEach(function(value, key) {
                if (result.score > 0) return; // Already found gibberish

                // Skip internal fields
                if (key.startsWith('_oh_') || key.startsWith('_wp')) return;

                // Skip email and phone fields
                if (/email|phone|tel|mail|fax/i.test(key)) return;

                // Check if this field should be examined
                var shouldCheck = checkFields.some(function(field) {
                    return key.toLowerCase().indexOf(field) !== -1;
                });

                if (!shouldCheck) return;

                var text = (typeof value === 'string') ? value.trim() : '';
                if (text.length < 6) return;

                // Skip if looks like valid data
                if (/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(text)) return; // Email
                if (/^https?:\/\//.test(text)) return; // URL
                if (/^[\d\s\+\-\(\)]+$/.test(text)) return; // Phone number

                if (self.isObviousGibberish(text)) {
                    result.score = 85;
                    result.reasons.push('Gibberish detected in: ' + key);
                    result.details.gibberishField = key;
                    result.details.gibberishValue = text.substring(0, 50);
                }
            });

            return result;
        },

        // Check if text is obvious gibberish
        isObviousGibberish: function(text) {
            var condensed = text.replace(/\s+/g, '');
            var len = condensed.length;

            if (len < 6) return false;

            // Check 1: Very low character variety (keyboard mashing like "safasafafafaffff")
            var chars = condensed.toLowerCase().split('');
            var uniqueChars = chars.filter(function(c, i, arr) {
                return arr.indexOf(c) === i;
            }).length;
            var varietyRatio = uniqueChars / len;

            if (len >= 8 && varietyRatio < 0.35) {
                return true;
            }

            // Check 2: Repeated short patterns (like "asdfasdf", "qwer qwer")
            var lower = condensed.toLowerCase();
            if (len >= 8) {
                for (var patternLen = 2; patternLen <= 4; patternLen++) {
                    var pattern = lower.substring(0, patternLen);
                    var repeated = '';
                    for (var k = 0; k < Math.ceil(len / patternLen); k++) {
                        repeated += pattern;
                    }
                    if (repeated.substring(0, len) === lower) {
                        return true;
                    }
                }
            }

            // Check 3: Very low vowel ratio
            var vowelCount = (condensed.match(/[aeiouAEIOU]/g) || []).length;
            var vowelRatio = vowelCount / len;
            if (vowelRatio < 0.08 && len > 6) {
                return true;
            }

            // Check 4: Long consonant sequences (5+ consonants in a row)
            if (/[bcdfghjklmnpqrstvwxzBCDFGHJKLMNPQRSTVWXZ]{5,}/.test(condensed)) {
                return true;
            }

            // Check 5: Random mixed case pattern (like "QGNItWfBJmqYrUNcxjYOmdZ")
            var uppercase = (condensed.match(/[A-Z]/g) || []).length;
            var lowercase = (condensed.match(/[a-z]/g) || []).length;
            if (uppercase > 3 && lowercase > 3 && len > 10) {
                var midUpper = (condensed.match(/(?<=[a-z])[A-Z]/g) || []).length;
                if (midUpper > 3) {
                    return true;
                }
            }

            return false;
        },

        heuristicAnalysis: function(content) {
            var result = {
                score: 0,
                reasons: [],
                details: {}
            };

            // Link density check
            var linkMatches = content.match(/https?:\/\/[^\s]+/gi) || [];
            if (linkMatches.length > OpenHoneypotConfig.maxLinks) {
                result.score += 40;
                result.reasons.push('Too many links (' + linkMatches.length + ')');
                result.details.linkCount = linkMatches.length;
            }

            // Common spam patterns
            var spamPatterns = [
                { pattern: /\b(viagra|cialis|casino|lottery|winner|congratulations|you\s+won)\b/i, score: 30 },
                { pattern: /\b(crypto|bitcoin|investment\s+opportunity|make\s+money\s+fast)\b/i, score: 25 },
                { pattern: /\b(click\s+here|act\s+now|limited\s+time|free\s+offer)\b/i, score: 20 },
                { pattern: /\b(SEO\s+service|backlink|rank\s+#1|first\s+page\s+google)\b/i, score: 35 },
                { pattern: /\b(dear\s+sir|dear\s+madam|dear\s+friend|to\s+whom\s+it\s+may\s+concern)\b/i, score: 15 },
                { pattern: /(.)\1{4,}/, score: 15 }, // Repeated characters
                { pattern: /[A-Z]{10,}/, score: 10 } // Excessive caps
            ];

            spamPatterns.forEach(function(item) {
                if (item.pattern.test(content)) {
                    result.score += item.score;
                    result.reasons.push('Spam pattern detected');
                    result.details.patternMatch = true;
                }
            });

            // HTML/Script injection attempt
            if (/<[^>]*script|<[^>]*iframe|<[^>]*object/i.test(content)) {
                result.score += 50;
                result.reasons.push('Potential code injection');
                result.details.injection = true;
            }

            // Multiple email addresses
            var emailMatches = content.match(/[\w.-]+@[\w.-]+\.\w+/g) || [];
            if (emailMatches.length > 2) {
                result.score += 25;
                result.reasons.push('Multiple email addresses');
                result.details.emailCount = emailMatches.length;
            }

            return result;
        },

        getFormContent: function(formData) {
            var content = '';
            var excludeFields = ['_oh_timestamp', '_oh_token', OpenHoneypotConfig.honeypotFieldName];

            // Add additional honeypot names to exclude list
            if (OpenHoneypotConfig.additionalHoneypotNames) {
                excludeFields = excludeFields.concat(OpenHoneypotConfig.additionalHoneypotNames);
            }

            formData.forEach(function(value, key) {
                if (excludeFields.indexOf(key) === -1 && !key.startsWith('_oh_')) {
                    content += ' ' + value;
                }
            });

            return content.trim();
        },

        handleSpam: function(form, result) {
            this.log('Spam detected:', result);

            // Track blocked entries
            if (result.details.emailBlacklisted) {
                this.addBlockedEntry('email', result.details.emailBlacklisted, 'Email blacklisted');
            }
            if (result.details.domainBlacklisted) {
                this.addBlockedEntry('domain', result.details.domainBlacklisted, 'Domain blacklisted');
            }
            if (result.details.blacklistMatch) {
                this.addBlockedEntry('keyword', result.details.blacklistMatch, 'Keyword matched');
            }
            if (result.details.gibberishField) {
                this.addBlockedEntry('gibberish', result.details.gibberishValue, 'Gibberish in ' + result.details.gibberishField);
            }
            if (result.details.honeypot) {
                this.addBlockedEntry('honeypot', 'form_submission', 'Honeypot triggered');
            }

            // Increment spam count
            this.incrementSpamCount();

            // Show user-friendly error
            this.showError(form, OpenHoneypotConfig.errorMessage);

            // Check escalation
            this.checkEscalation();
        },

        showError: function(form, message) {
            // Remove existing error
            var existingError = form.querySelector('.oh-error-message');
            if (existingError) {
                existingError.remove();
            }

            var errorDiv = document.createElement('div');
            errorDiv.className = 'oh-error-message';
            errorDiv.style.cssText = 'background:#f8d7da;color:#721c24;padding:12px 16px;border-radius:4px;margin-bottom:16px;font-size:14px;border:1px solid #f5c6cb;';
            errorDiv.textContent = message;

            form.insertBefore(errorDiv, form.firstChild);

            // Auto-remove after 5 seconds
            setTimeout(function() {
                if (errorDiv.parentNode) {
                    errorDiv.remove();
                }
            }, 5000);
        },

        // Rate Limiting
        isRateLimited: function() {
            var now = Date.now();
            var windowMs = OpenHoneypotConfig.rateLimitWindow * 1000;

            // Filter to keep only timestamps within window
            this.submissionTimestamps = this.submissionTimestamps.filter(function(ts) {
                return (now - ts) < windowMs;
            });

            return this.submissionTimestamps.length >= OpenHoneypotConfig.rateLimitSubmissions;
        },

        trackSubmission: function() {
            this.submissionTimestamps.push(Date.now());
            this.saveSubmissionTimestamps();
        },

        loadSubmissionTimestamps: function() {
            try {
                var stored = sessionStorage.getItem('oh_submissions');
                if (stored) {
                    var now = Date.now();
                    var windowMs = OpenHoneypotConfig.rateLimitWindow * 1000;

                    this.submissionTimestamps = JSON.parse(stored).filter(function(ts) {
                        return (now - ts) < windowMs;
                    });
                }
            } catch (e) {
                // Session storage not available
            }
        },

        saveSubmissionTimestamps: function() {
            try {
                sessionStorage.setItem('oh_submissions', JSON.stringify(this.submissionTimestamps));
            } catch (e) {
                // Session storage not available
            }
        },

        // Spam count tracking
        incrementSpamCount: function() {
            var now = Date.now();
            this.spamTimestamps.push(now);

            // Keep only timestamps within the window
            var windowMs = OpenHoneypotConfig.escalationWindow * 60 * 1000;
            this.spamTimestamps = this.spamTimestamps.filter(function(ts) {
                return (now - ts) < windowMs;
            });

            this.spamCount = this.spamTimestamps.length;

            // Save to session storage
            try {
                sessionStorage.setItem('oh_spam_timestamps', JSON.stringify(this.spamTimestamps));
            } catch (e) {
                // Session storage not available
            }
        },

        loadSpamCount: function() {
            try {
                var stored = sessionStorage.getItem('oh_spam_timestamps');
                if (stored) {
                    var now = Date.now();
                    var windowMs = OpenHoneypotConfig.escalationWindow * 60 * 1000;

                    this.spamTimestamps = JSON.parse(stored).filter(function(ts) {
                        return (now - ts) < windowMs;
                    });

                    this.spamCount = this.spamTimestamps.length;
                }
            } catch (e) {
                // Session storage not available
            }
        },

        checkEscalation: function() {
            if (!OpenHoneypotConfig.escalationEnabled) {
                return;
            }

            if (this.spamCount >= OpenHoneypotConfig.escalationThreshold) {
                this.showEscalationBanner();
            }
        },

        showEscalationBanner: function() {
            // Check if banner already exists
            if (document.querySelector('.oh-escalation-banner')) {
                return;
            }

            var banner = document.createElement('div');
            banner.className = 'oh-escalation-banner';
            banner.style.cssText = 'position:fixed;bottom:20px;right:20px;background:linear-gradient(135deg,#2c5aa0 0%,#1e3d6b 100%);color:#fff;padding:16px 24px;border-radius:8px;box-shadow:0 4px 20px rgba(0,0,0,0.2);z-index:9999;max-width:320px;font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;';

            banner.innerHTML = '<div style="display:flex;align-items:flex-start;gap:12px;">' +
                '<span style="font-size:24px;">&#9888;</span>' +
                '<div>' +
                '<strong style="display:block;margin-bottom:4px;">Experiencing Spam Issues?</strong>' +
                '<p style="margin:0 0 12px 0;font-size:13px;opacity:0.9;">We can help you implement advanced protection.</p>' +
                '<a href="' + OpenHoneypotConfig.supportUrl + '" target="_blank" style="display:inline-block;background:#fff;color:#2c5aa0;padding:8px 16px;border-radius:4px;text-decoration:none;font-weight:600;font-size:13px;">Get Expert Help</a>' +
                '</div>' +
                '<button onclick="this.parentElement.parentElement.remove()" style="background:none;border:none;color:#fff;cursor:pointer;font-size:20px;line-height:1;padding:0;opacity:0.8;">&times;</button>' +
                '</div>';

            document.body.appendChild(banner);
        },

        // Utilities
        generateToken: function() {
            var array = new Uint8Array(16);
            if (window.crypto && window.crypto.getRandomValues) {
                window.crypto.getRandomValues(array);
            } else {
                for (var i = 0; i < 16; i++) {
                    array[i] = Math.floor(Math.random() * 256);
                }
            }
            return Array.from(array, function(byte) {
                return byte.toString(16).padStart(2, '0');
            }).join('');
        },

        log: function() {
            if (OpenHoneypotConfig.debug) {
                console.log.apply(console, ['[Open Honeypot]'].concat(Array.prototype.slice.call(arguments)));
            }
        },

        // =========================================================================
        // BLOCKED ENTRIES MANAGEMENT (localStorage-based)
        // Since client-side JS cannot access visitor IPs, we track blocked emails
        // =========================================================================

        /**
         * Add an entry to the blocked list
         */
        addBlockedEntry: function(type, value, reason) {
            var entries = this.getBlockedEntries();
            var entry = {
                type: type, // 'email', 'keyword', 'gibberish', etc.
                value: value,
                reason: reason,
                timestamp: Date.now(),
                blocked: true
            };

            // Check if already exists
            var exists = entries.some(function(e) {
                return e.type === type && e.value === value;
            });

            if (!exists) {
                entries.push(entry);
                this.saveBlockedEntries(entries);
                this.log('Blocked entry added:', entry);
            }
        },

        /**
         * Get all blocked entries
         */
        getBlockedEntries: function() {
            try {
                var stored = localStorage.getItem('oh_blocked_entries');
                return stored ? JSON.parse(stored) : [];
            } catch (e) {
                return [];
            }
        },

        /**
         * Save blocked entries
         */
        saveBlockedEntries: function(entries) {
            try {
                localStorage.setItem('oh_blocked_entries', JSON.stringify(entries));
            } catch (e) {
                this.log('Failed to save blocked entries:', e);
            }
        },

        /**
         * View all blocked entries (console-friendly)
         */
        viewBlockedEntries: function() {
            var entries = this.getBlockedEntries();
            if (entries.length === 0) {
                console.log('[Open Honeypot] No blocked entries found.');
                return [];
            }

            console.log('[Open Honeypot] Blocked Entries (' + entries.length + '):');
            console.table(entries.map(function(e, i) {
                return {
                    index: i,
                    type: e.type,
                    value: e.value,
                    reason: e.reason,
                    date: new Date(e.timestamp).toLocaleString(),
                    blocked: e.blocked
                };
            }));

            return entries;
        },

        /**
         * Unblock an entry by index
         */
        unblockEntry: function(index) {
            var entries = this.getBlockedEntries();
            if (index < 0 || index >= entries.length) {
                console.log('[Open Honeypot] Invalid index. Use viewBlockedEntries() to see all entries.');
                return false;
            }

            var entry = entries[index];
            entry.blocked = false;
            this.saveBlockedEntries(entries);
            console.log('[Open Honeypot] Unblocked entry:', entry.type, '-', entry.value);
            return true;
        },

        /**
         * Remove an entry completely
         */
        removeEntry: function(index) {
            var entries = this.getBlockedEntries();
            if (index < 0 || index >= entries.length) {
                console.log('[Open Honeypot] Invalid index. Use viewBlockedEntries() to see all entries.');
                return false;
            }

            var removed = entries.splice(index, 1)[0];
            this.saveBlockedEntries(entries);
            console.log('[Open Honeypot] Removed entry:', removed.type, '-', removed.value);
            return true;
        },

        /**
         * Clear all blocked entries
         */
        clearBlockedEntries: function() {
            try {
                localStorage.removeItem('oh_blocked_entries');
                console.log('[Open Honeypot] All blocked entries cleared.');
                return true;
            } catch (e) {
                return false;
            }
        },

        /**
         * Check if a value is blocked
         */
        isBlocked: function(type, value) {
            var entries = this.getBlockedEntries();
            return entries.some(function(e) {
                return e.type === type && e.value === value && e.blocked;
            });
        },

        /**
         * Export blocked entries as JSON
         */
        exportBlockedEntries: function() {
            var entries = this.getBlockedEntries();
            var json = JSON.stringify(entries, null, 2);
            console.log('[Open Honeypot] Blocked Entries JSON:');
            console.log(json);
            return json;
        },

        /**
         * Import blocked entries from JSON
         */
        importBlockedEntries: function(json) {
            try {
                var entries = JSON.parse(json);
                if (!Array.isArray(entries)) {
                    throw new Error('Invalid format');
                }
                this.saveBlockedEntries(entries);
                console.log('[Open Honeypot] Imported ' + entries.length + ' entries.');
                return true;
            } catch (e) {
                console.log('[Open Honeypot] Failed to import:', e.message);
                return false;
            }
        }
    };

    // Initialize
    OpenHoneypot.init();

    // Expose for external access if needed
    window.OpenHoneypot = OpenHoneypot;
    window.OpenHoneypotConfig = OpenHoneypotConfig;

})();
